/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package SPITServerAutoDetectPackage;

import java.awt.EventQueue;
import java.util.ArrayList;

/**
 * This EventQueueHandler collects Runnables, which will be normally send to the EventQueue. 
 * The Runnables will be added to a ArrayList when invokeLater has been called. 
 * An extra Thread catches one Runnable after the other and calls EventQueue.invokeLater(p_Runnable) (fifo).
 * This technic is necessary, because the invokeLater function of the EventQueue blocks while performing a Runnable.<br>
 * If a Runnable is still in the Runnable-list it will not be added twice - to increase the invokeLater calls.<br>
 * @author Hagi
 * @see EventQueue
 */
public class EventQueueHandler {
	private static ArrayList<Runnable>		m_Runnables = new ArrayList<Runnable>();

	private static Thread					m_HandlerThread;
	private static boolean					m_HandlerStop = false;
	private static final Object				m_SyncRunnables = new Object();


	/**
	 * Adds a Runnable to the list of Runnables which should be invoked later by the EVentQueue.<br>
	 * This function returns immediately.<br
	 * An extra Thread will call EventQueue.invokeLater(p_Runnable) for each Runnable in the list (fifo).<br>
	 * @param p_Runnable The Runnable instance which should be added. 
	 * If the Runnable instance is still in the list it will not be added twice.<br>
	 * @see EventQueue#invokeLater(java.lang.Runnable) 
	 */
	public static void invokeLater(final Runnable p_Runnable) {
		if (m_HandlerThread == null) {
			Runnable o_RunnableHandler = new Runnable() {
				@Override
				public void run() {
					handleInvokeLater();
				}
			};
			m_HandlerThread = new Thread(null, o_RunnableHandler, EventQueueHandler.class.getSimpleName());
			m_HandlerThread.start();
		}
		
		synchronized(m_SyncRunnables) {
			if (m_Runnables.contains(p_Runnable)) {
				return;
			}
			m_Runnables.add(p_Runnable);
			m_SyncRunnables.notifyAll();
		}

	}

	/**
	 * Removes a Runnable from the list of Runnables which should be invoked later.<br>
	 * @param p_Runnable 
	 */
	public static void removeRunnable(Runnable p_Runnable) {
		synchronized(m_SyncRunnables) {
			m_Runnables.remove(p_Runnable);
		}
	}

	private static void handleInvokeLater() {
		Runnable o_Runnable;
		while (m_HandlerStop == false) {
			o_Runnable = null;
			synchronized(m_SyncRunnables) {
				if (m_Runnables.size() <= 0) {
					try {
						m_SyncRunnables.wait();
						continue;
					}
					catch (Exception e) {

					}
				}
				else {
					o_Runnable = m_Runnables.get(0);
					m_Runnables.remove(0);
				}
			}
			if (o_Runnable == null) continue;
			EventQueue.invokeLater(o_Runnable);
		}
	}


	/**
	 * This function will stop the extra Thread sending the Runnables to the EventQueue.<br>
	 * All Runnables from the list will be removed.<br>
	 */
	public static void close() {
		synchronized(m_SyncRunnables) {
			m_HandlerStop = true;
			m_SyncRunnables.notifyAll();
		}
		try {
			if (Thread.currentThread() != m_HandlerThread) {
				m_HandlerThread.join(2000);
			}
		}
		catch (Exception e) {
		}
		m_Runnables.clear();
		
	}
	
	/**
	 * This will stop the execuing thread, clears the kist of Runnables and restarts the Thread.<br>
	 */
	public static void interrupt() {

		synchronized(m_SyncRunnables) {
			m_HandlerStop = true;
			m_SyncRunnables.notifyAll();
		}
		
		try {
			if (Thread.currentThread() != m_HandlerThread && m_HandlerThread != null) {
				m_HandlerThread.join();
			}
		}
		catch (Exception e) {
		}
		m_Runnables.clear();
		
		m_HandlerStop = false;
		
		Runnable o_RunnableHandler = new Runnable() {
			@Override
			public void run() {
				handleInvokeLater();
			}
		};
		m_HandlerThread = new Thread(null, o_RunnableHandler, EventQueueHandler.class.getSimpleName());
		m_HandlerThread.start();		
	}
	
	
}
