/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package SPITClientPackage;

import SPITLibraryPackage.SPIT_ClientInterface;
import SPITLibraryPackage.SPIT_Message;
import java.awt.EventQueue;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketTimeoutException;

/**
 * The socket processes the DatagramSocket connection to the SPIT_Server.<br>
 * The communication between SPIT_Client and SPIT_Server is via a UDP network connection,
 * @author Hagi
 */
public class SPIT_ClientSocket {
	public static boolean	s_DEBUGON = true; 
	private SPIT_Client		m_SPITClient;
	private DatagramSocket	m_DatagramSocket;
	private InetAddress		m_ServerInetAddress;
	private int				m_ServerPort;
	private Runnable		m_RunnableReceive;
	private boolean			m_StopReceive;
	private Thread			m_ThreadReceive;
	private final Object	m_Sync = new Object();
	

	public SPIT_ClientSocket(SPIT_Client p_SPITClient) {
		initMembers();
		m_SPITClient = p_SPITClient;
	}
	private void initMembers() {
		m_DatagramSocket = null;
		m_ServerInetAddress = null;
		m_ServerPort = 0;
		//initialize listening to DatagramSocket
		initRunnableReceive();
	}
	/**
	 * 
	 * @param p_ServerInetAddress InetAddress of the SPIT_Server
	 * @param p_ServerPort Port number of the SPIT_Server
	 * @param p_ClientPort port number of the SPIT_Client<br>
	 * maybe 0 then any free port number of the local machine is used
	 * @param p_LoginName login name, is checked by the server when the SPITConnect message is sent 
	 * @param p_LoginPassword login password, is checked by the server when the SPITConnect message is sent 
	 * @return true = The SPIT_ClientSocket can listen to incoming UDP packets and the SPITConnect message was sended<br>
	 * false = the p_ServerInetAddress == null or<br>
	 * p_ServerPort is > 0 and the DatagramSocket could not be bound to this port number or <br>
	 * the SPITConnect message could not be send or <br>
	 * login parameters are wrong
	 */
	public boolean connect(InetAddress p_ServerInetAddress, int p_ServerPort, int p_ClientPort, String p_LoginName, String p_LoginPassword) {
		if (p_ServerInetAddress == null) return false;
		if (p_ServerPort < 0 || p_ServerPort > 65535) return false;
		if (p_ClientPort > 65535) return false;
		
		
		SPIT_Message o_SPIT_Message;
		SPIT_ClientInterface o_SPIT_Client;
		
		//Disconnect prior connection if it exists
		disconnect();
		
		o_SPIT_Client = m_SPITClient;
		if (o_SPIT_Client == null) return false;
		
		try {
			m_ServerInetAddress = p_ServerInetAddress;
			m_ServerPort = p_ServerPort;
			if (p_ClientPort <= 0) {
				//create a datagrem socket bind to any port on the local machine.
				//The socket will be bound to the wildcard address
				m_DatagramSocket = new DatagramSocket();
			}
			else {
				m_DatagramSocket = new DatagramSocket(p_ServerPort);
			}
		}
		catch (Exception e) {
			m_ServerInetAddress = null;
			m_ServerPort = 0;
			m_DatagramSocket = null;
			System.out.println("SPIT_ClientSocket connect ERROR: " + e.getMessage());
			return false;
		}
		try {
			m_DatagramSocket.setSendBufferSize(131072);
		}
		catch (Exception e) {
			
		}
		//Listen to DatagramSocket
		m_StopReceive = false;
		m_ThreadReceive = new Thread(m_RunnableReceive);
		m_ThreadReceive.start();
		o_SPIT_Message = SPIT_Message.createConnect(o_SPIT_Client.getClientComputerName(), o_SPIT_Client.getName(), 
				p_LoginName, p_LoginPassword, 
				SPIT_Client.cleanIPAddress(m_ServerInetAddress.getHostAddress()), m_ServerPort);
		//Don't send the mesaage direct via sendSPIT_Message(..)
		//because the header params must be set, this will be done in SPIT_Client.sendMessage(..)
		return SPIT_Client.sendMessage(o_SPIT_Message);
	}
	
	/**
	 * Disconnect from SPIT_Server
	 * @return true = the SPITDisconnect message could be sent to the server<br>
	 * false = the SPITDisconnect message could NOT be sent to the server<br>
	 */
	public boolean disconnect() {
		DatagramSocket o_DatagramSocket;
		boolean boolOK;
		m_StopReceive = true;
		o_DatagramSocket = m_DatagramSocket;
		if (o_DatagramSocket == null) return true;
		if (o_DatagramSocket.isClosed()) {
			m_DatagramSocket = null;
			return true;
		}
		
		boolOK = sendDisconnect();
		
		o_DatagramSocket.close();
		
		m_DatagramSocket = null;
		return boolOK;
	}
	
	private void initRunnableReceive() {
		m_RunnableReceive = new Runnable() {
			@Override
			public void run() {
				byte[] o_BytesReceived;
				DatagramPacket o_DatagramPacket;
				o_BytesReceived = new byte[512];
				SPIT_Message o_SPIT_Message;
				SPIT_Message.OPPart o_SPIT_Message_OPPart;
				SPIT_Message.Header o_SPIT_Message_Header;
				SPIT_Client o_SPITClient;
				while (m_StopReceive == false) {
					synchronized (m_Sync) {
						if (m_DatagramSocket == null || m_DatagramSocket.isClosed()) {
							m_StopReceive = true;
							continue;
						}
					}
					o_DatagramPacket = new DatagramPacket(o_BytesReceived, o_BytesReceived.length);
					try {
						m_DatagramSocket.receive(o_DatagramPacket);
					}
					catch (SocketTimeoutException e_SocketTimeoutException) {
						o_DatagramPacket = null;
						continue;
					}
					catch (Exception e) {
						m_StopReceive = true;
						o_DatagramPacket = null;
					}
						
					if (o_DatagramPacket == null) {
						continue;
					}
					o_SPITClient = m_SPITClient;
					
					//parseMessage: try to fetch the header of SPIT_Message
					o_SPIT_Message = new SPIT_Message(o_DatagramPacket.getData());
					o_SPIT_Message_Header = o_SPIT_Message.getHeader();
					if (o_SPIT_Message_Header == null) {
						//the message was no SPIT_Message
						//or the header is corrupted, perhaps wrong SPIT version 
						if (s_DEBUGON) {
							System.out.println("SPIT_ClientSocket RunnableReceive ERROR SPIT_Message_Header is NULL");
						}
						o_SPIT_Message = null;
						continue;
					}
					//parseMessage: try to fetch the OP_Part of SPIT_Message
					o_SPIT_Message_OPPart = o_SPIT_Message.getOPPart();
					if (o_SPIT_Message_OPPart == null) {
						//the OPPart is coruppted, perhaps wrong SPIT version
						if (s_DEBUGON) {
							System.out.println("Error SPIT_ClientSocket RunnableReceive ERROR SPIT_Message_OPPart is NULL");
						}
						o_SPIT_Message = null;
						continue;
					}
						
					if (o_SPITClient != null) {
						if (s_DEBUGON) {
							System.out.println("<- Client receiveSPIT_Message OP " + o_SPIT_Message.getOPPart().getClass().getSimpleName());
						}
						o_SPITClient.parseSPIT_Message(o_SPIT_Message);
					}
				}

			}
		};
	}
	
	/**
	 * Returns the port number of the SPIT_ClientSocket (DatagramSocket)
	 * @return port number
	 */
	public int getPort() {
		DatagramSocket o_DatagramSocket;
		o_DatagramSocket = m_DatagramSocket;
		if (o_DatagramSocket == null) return 0;
		return o_DatagramSocket.getPort();
	}
	/**
	 * Returns the IP address of the SPIT_ClientSocket (DatagramSocket)
	 * @return ip address
	 */
	public String getIPAddress() {
		InetAddress o_InetAddress;
		DatagramSocket o_DatagramSocket;
		String stringIPAddress;
		o_DatagramSocket = m_DatagramSocket;
		if (o_DatagramSocket == null) return "";
		o_InetAddress = o_DatagramSocket.getInetAddress();
		if (o_InetAddress == null) return "";
		stringIPAddress = SPIT_Client.cleanIPAddress(o_InetAddress.getHostAddress());
		return stringIPAddress;
	}
	
	public InetAddress getInetAddress() {
		InetAddress o_InetAddress;
		DatagramSocket o_DatagramSocket;
		o_DatagramSocket = m_DatagramSocket;
		if (o_DatagramSocket == null) return null;
		o_InetAddress = o_DatagramSocket.getInetAddress();
		return o_InetAddress;
	}
	/**
	 * Send a SPIT_Message to the SPIT_Server.<br>
	 * The SPIT_Server ip adsress and port number is set prior by connect 
	 * @param p_SPIT_Message
	 * @return true = the message was send<br>
	 * false = the message could not be send
	 */
	public boolean sendSPIT_Message(SPIT_Message p_SPIT_Message) {
		if (p_SPIT_Message == null) return false;
		DatagramSocket o_DatagramSocket;
		DatagramPacket o_DatagramPacket;
		byte[] o_MessageBytes;
		
		o_DatagramSocket = m_DatagramSocket;
		if (o_DatagramSocket == null) return false;
		o_MessageBytes = p_SPIT_Message.getMessageBytes();
		if (o_MessageBytes == null) return false;
		if (s_DEBUGON) {
			System.out.println("-> Client sendSPIT_Message OP " + p_SPIT_Message.getOPPart().getClass().getSimpleName());
		}
		o_DatagramPacket = new DatagramPacket(o_MessageBytes, o_MessageBytes.length, m_ServerInetAddress, m_ServerPort);
		try {
			o_DatagramSocket.send(o_DatagramPacket);
		}
		catch (Exception e) {
			System.out.println("SPIT_ClientSocket sendSPIT_Message ERROR " + e.getMessage());
			return false;
		}
		return true;
	}
	
	/**
	 * Send the SPITDisconnect message
	 * @return true = the message was send<br>
	 * false = the connection is not established or the message could not be send
	 */
	private boolean sendDisconnect() {
		if (m_SPITClient == null) return false;
		SPIT_Message o_SPIT_Message;
		o_SPIT_Message = SPIT_Message.createDisconnect();
		//Don't send the mesaage direct via sendSPIT_Message(..)
		//because the header params must be set, this will be done in SPIT_Client.sendMessage(..)
		return SPIT_Client.sendMessage(o_SPIT_Message);
	}
	
	/**
	 * Returns the InetAddress of the SPIT_Server (Server).
	 * This has been set by connect(..)
	 * @return the InetAddress of the SPIT_Server
	 */
	public InetAddress getServerInetAddress() {
		return m_ServerInetAddress;
	}
	/**
	 * Returns the IP Address of the SPIT_Server (Server).
	 * This has been set by connect(..)
	 * @return the IP of the SPIT_Server
	 */
	public String getServerHostAddress() {
		if (m_ServerInetAddress == null) return "";
		String stringIPAddress;
		stringIPAddress = SPIT_Client.cleanIPAddress(m_ServerInetAddress.getHostAddress());
		return stringIPAddress;
	}
	/**
	 * Returns the port number of the SPIT_Server (Server).
	 * This has been set by connect(..)
	 * @return the port number of the SPIT_Server
	 */
	public int getServerPort() {
		return m_ServerPort;
	}
}
