/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

/* 
 * Windows implementation of a SPITSocket.
 * 
 * File:   WindowsSocket.cpp
 * Author: Hagi
 * 
 * Created on 11. April 2019, 08:36
 */





#include "_SPITMACROS.h"
#if defined(OS_WINDOWS)
#include "WindowsSocket.h"


//----- -----
WindowsSocket::WindowsSocket() : SPITSocket(){
	SPITSocket::FAMILY_IP4 = AF_INET;
	SPITSocket::FAMILY_IP6 = AF_INET6;
	m_UDPSocketIP4 = INVALID_SOCKET;
	m_UDPSocketIP6 = INVALID_SOCKET;
	createSocket_UDP();
}


WindowsSocket::~WindowsSocket() {
	
	int intResultIP4;
	int intResultIP6;
	if (m_UDPSocketIP4 != INVALID_SOCKET) {
//		intResultIP4 = shutdown(m_UDPSocketIP4, 2);
		intResultIP4 = closesocket(m_UDPSocketIP4);
		m_UDPSocketIP4 = INVALID_SOCKET;
	}
	if (m_UDPSocketIP6 != INVALID_SOCKET) {
//		intResultIP6 = shutdown(m_UDPSocketIP6, 2);
		intResultIP6 = closesocket(m_UDPSocketIP6);
		m_UDPSocketIP6 = INVALID_SOCKET;
	}
	this_thread::sleep_for(chrono::milliseconds(100));
}
/**
 * Initialize WSAStartup
 * @return 
 */
int WindowsSocket::init() {
  WSADATA wsa;
  int intResult;
  intResult = WSAStartup(MAKEWORD(2,2),&wsa);
  return intResult;
}
/**
 * Clean WSACleanup
 * @return 
 */
int WindowsSocket::cleanUp() {
	int intResult;
	intResult = WSACleanup();
	return intResult;
}

/**
 * Creates two new sockets.<br>
 * One socket for IP4 and one for IP6.<br>
 * If there exist some, they will be closed.<br>
 * @return 
 */
bool WindowsSocket::createSocket_UDP() {
	bool boolSuccess = true;
	int intResult = -1;
	int intResultNonBlocking = 0;
	BOOL o_OptionValue = TRUE;
	
	//ip4
	if (m_UDPSocketIP4 != INVALID_SOCKET) {
		closesocket(m_UDPSocketIP4);
		m_UDPSocketIP4 = INVALID_SOCKET;
	}
	if (m_UDPSocketIP6 != INVALID_SOCKET) {
		closesocket(m_UDPSocketIP6);
		m_UDPSocketIP6 = INVALID_SOCKET;
	}

	
	m_UDPSocketIP4 = socket(AF_INET,SOCK_DGRAM, 0);
	if (m_UDPSocketIP4 == INVALID_SOCKET) {
		boolSuccess = false;
	}
	else {
		intResult =  setsockopt(m_UDPSocketIP4, SOL_SOCKET, SO_REUSEADDR, (char*)&o_OptionValue, sizeof(o_OptionValue));

	}
	
	m_UDPSocketIP6 = socket(AF_INET6,SOCK_DGRAM,0);
	if (m_UDPSocketIP6 == INVALID_SOCKET) {
		boolSuccess = false;
		closesocket(m_UDPSocketIP4);
		m_UDPSocketIP4 = INVALID_SOCKET;
	}
	else {
		intResult =  setsockopt(m_UDPSocketIP6, SOL_SOCKET, SO_REUSEADDR, (char*)&o_OptionValue, sizeof(o_OptionValue));
	}
	if (intResult < 0) {
		boolSuccess = false;
	}
	if (boolSuccess == true) {
		//increase the buffer size of the socket
		int intBufferSize = 2046*512;
		intResult = setsockopt(m_UDPSocketIP4, SOL_SOCKET, SO_RCVBUF, (char*)&intBufferSize, sizeof(intBufferSize));
		intResult = setsockopt(m_UDPSocketIP4, SOL_SOCKET, SO_SNDBUF, (char*)&intBufferSize, sizeof(intBufferSize));
		intResult = setsockopt(m_UDPSocketIP6, SOL_SOCKET, SO_RCVBUF, (char*)&intBufferSize, sizeof(intBufferSize));
		intResult = setsockopt(m_UDPSocketIP6, SOL_SOCKET, SO_SNDBUF, (char*)&intBufferSize, sizeof(intBufferSize));
	}
	if (boolSuccess == true) {
		//set the m_UDPSocketIP4 and m_UDPSocketIP6 in NON BLOCKING mode because
		//if the UDPSockets are in blocking mode the receiveMessage does not return in every constellation  
		u_long blockingMode = 1; //1 = non blocking, 0 = blocking
		intResultNonBlocking = ioctlsocket(m_UDPSocketIP4, FIONBIO, &blockingMode);
		intResultNonBlocking = ioctlsocket(m_UDPSocketIP6, FIONBIO, &blockingMode);
		if (intResultNonBlocking != 0) {
			boolSuccess = false;
		}
	}
	return boolSuccess;
}

/**
 * Binds the socket to the client's ip address and port.<br>
 * The socket option SO_REUSEADDR is set to true.<br>
 * @param p_ClientIP ip address of the client
 * @param p_ClientPort port of the client.<br>
 * if p_Port == "0" any free port will be used<br>
 * @return -1 id bind fails, o if bind is ok
 * 
 */
int WindowsSocket::bindSocket(string p_ServerIP, string p_ClientIP, string p_ClientPort)
{
    ADDRINFO *clientAI;
	SOCKADDR_IN o_IP4SocketAddress;
	SOCKADDR_IN6 o_IP6SocketAddress;
	int socklen;
	string stringClientPort;
	if (p_ClientPort.length() <= 0) stringClientPort = "0";
	else stringClientPort = p_ClientPort;
	//int intResult = SOCKET_ERROR;
	int intResult = 0;
	if (getaddrinfo(p_ClientIP.data(), stringClientPort.data(), NULL, &clientAI) != 0) {
		return SOCKET_ERROR;
	}
	BOOL o_OptionValue = TRUE;
	switch(clientAI->ai_family) {
		case AF_INET:
			if (m_UDPSocketIP4 == INVALID_SOCKET) {
				break;
			}
			intResult = bind(m_UDPSocketIP4, (const sockaddr*)clientAI->ai_addr, (int)clientAI->ai_addrlen);
			if (intResult < 0) {
				break;
			}
			setBoundClientFamily(AF_INET);
			if (intResult == 0) {
				socklen = sizeof (o_IP4SocketAddress);
				if (getsockname(m_UDPSocketIP4, (SOCKADDR*)&o_IP4SocketAddress, &socklen) != -1) {
					setBoundClientPort(to_string((int)o_IP4SocketAddress.sin_port));
				}
			}
			break;
		case AF_INET6:
			if (m_UDPSocketIP6 == INVALID_SOCKET) {
				break;
			}
			if (intResult < 0) {
				break;
			}		
			intResult = bind(m_UDPSocketIP6, (const sockaddr*)clientAI->ai_addr, (int)clientAI->ai_addrlen);
			if (intResult < 0) {
				break;
			}
			setBoundClientFamily(AF_INET6);
			
			if (intResult == 0) {
				socklen = sizeof (o_IP6SocketAddress);
				if (getsockname(m_UDPSocketIP6, (SOCKADDR*)&o_IP6SocketAddress, &socklen) != -1) {
					setBoundClientPort(to_string((int)o_IP6SocketAddress.sin6_port));
				}				
			}
			break;
		default:
			setBoundClientFamily(0);
			intResult = SOCKET_ERROR;
			break;
	}
	freeaddrinfo(clientAI);

	if (intResult == SOCKET_ERROR) {
		setIsBound(false);
	}
	else {
		setIsBound(true);
	}
	return intResult;
}

/**
 * Set the sockopt SO_BROADCAST to 0 or 1
 * @param p_Broadcast true SO_BROADCAST is set<br>
 * false = SO_BROADCAST is not set
 * @return 0 = ok, -1 = error
 */
int WindowsSocket::setBroadcast(bool p_Broadcast) {
	int intResult;
	//IP4
	BOOL  o_BroadcastValue = FALSE;
	if (p_Broadcast) {
		o_BroadcastValue = TRUE;
	}
	else {
		o_BroadcastValue = FALSE;
	}
    intResult = setsockopt(m_UDPSocketIP4, SOL_SOCKET, SO_BROADCAST, (char*)&o_BroadcastValue, sizeof(BOOL));
	
	//IP6
	DWORD loop = 1;
	if (p_Broadcast) {
		loop = 1;
	}
	else {
		loop = 0;
	}
	intResult = setsockopt(m_UDPSocketIP6, IPPROTO_IPV6, IPV6_MULTICAST_LOOP, (char*)&loop, sizeof (loop));
	return intResult;
}
/**
 * 
 * @param p_InetFamily AF_INET or AF_INET6
 * @param p_ServerIP the ip address of the server
 * @param p_ServerPort the port of the server
 * @param p_MessageBytes buffer containing the message 
 * @param p_MessageLength the length of the message
 * @return -1 send fails<br>
 * 0 send is done
 */
int WindowsSocket::sendMessageTo(int p_InetFamily, string p_ServerIP, string p_ServerPort, const char* p_MessageBytes, int p_MessageLength) {
	if (m_UDPSocketIP4 == INVALID_SOCKET && m_UDPSocketIP6 == INVALID_SOCKET) return -1;
	ADDRINFO *ptrAddressInfo;

	int intResult = SOCKET_ERROR;
	// prepare AddressInfo
	if (getaddrinfo(p_ServerIP.data(), p_ServerPort.data(), NULL, &ptrAddressInfo) != 0) {
	    return -1;
	}
	switch(ptrAddressInfo->ai_family) {
		case AF_INET:
			if (m_UDPSocketIP4 != INVALID_SOCKET) {
				intResult = sendto(m_UDPSocketIP4, p_MessageBytes, p_MessageLength, 0, ptrAddressInfo->ai_addr, ptrAddressInfo->ai_addrlen);
				freeaddrinfo(ptrAddressInfo);
			}
			break;
		case AF_INET6:
			if (m_UDPSocketIP6 != INVALID_SOCKET) {		
				intResult = sendto(m_UDPSocketIP6, p_MessageBytes, p_MessageLength, 0, ptrAddressInfo->ai_addr, ptrAddressInfo->ai_addrlen);
				freeaddrinfo(ptrAddressInfo);
			}
			break;
		default:
			intResult = -1;
	}
	if (intResult == SOCKET_ERROR) {
		return -1;
	}
	return 0;
}
/**
 * Starts receiving a message
 * @param p_InetFamily	AF_INET or AF_INET6
 * @param p_Buffer	a char* buffer large enough to hold the expected data 
 * @param p_BufferLength the length of the buffer
 * @param p_ServerIP = NULL, the ip address is not determined<br>
 * != NULL after receiving a message p_ServerID contains the ip address of the sender/server
 * @return 
 */
int WindowsSocket::receiveMessage(int p_InetFamily, char* p_Buffer, int p_BufferLength, string* p_ServerIP) {
	int intResult = SOCKET_ERROR;
//	ADDRINFO *AI;
	SOCKADDR_IN o_IP4SocketAddress;
	SOCKADDR_IN6 o_IP6SocketAddress;
	int socklen;
	switch(p_InetFamily) {
		case AF_INET:
			if (m_UDPSocketIP4 == INVALID_SOCKET) {
				break;
			}
			socklen = sizeof (o_IP4SocketAddress);
			intResult = recvfrom(m_UDPSocketIP4, p_Buffer, p_BufferLength, 0, (SOCKADDR *)&o_IP4SocketAddress, &socklen);
			if (intResult >= 0) {
				if (p_ServerIP != NULL) {
					*p_ServerIP = inet_ntoa(o_IP4SocketAddress.sin_addr);
				}
			}
			break;
		case AF_INET6:
			if (m_UDPSocketIP6 == INVALID_SOCKET) {
				break;
			}
			socklen = sizeof (o_IP6SocketAddress);
			intResult = recvfrom(m_UDPSocketIP6, p_Buffer, p_BufferLength, 0, (SOCKADDR *)&o_IP6SocketAddress, &socklen);
			if (intResult >= 0) {
				if (p_ServerIP != NULL) {
					char clienthost[NI_MAXHOST];  //The clienthost will hold the IP address.
					char clientservice[NI_MAXSERV];
					GetNameInfo((SOCKADDR *)&o_IP6SocketAddress, socklen, clienthost, sizeof(clienthost), clientservice, sizeof(clientservice), NI_NUMERICHOST|NI_NUMERICSERV);
					*p_ServerIP = clienthost
					//perhaps the sin_Scope_id is not added to p_ServerIP (xxxxxx%scope_id)
					//we have to add "%" + o_IP6SocketAddress.sin_scope_id
				}
			}
			break;
	}
	return intResult;
}

/**
 * Calculates the broadcast address from a ip4 address.<br>
 * @param p_IPAddress the ip4 address
 * @return returns the broadcast address
 */
string WindowsSocket::getBroadcastIP4(SPITSocket::IPAddress* p_IPAddress) {
    struct in_addr broadcast;
	unsigned long ulIPAddress;
	unsigned long ulSubnetMask;
	
	ulIPAddress = inet_addr(p_IPAddress->getIPAddress().data());
	ulSubnetMask = inet_addr(p_IPAddress->getSubnetMask().data());
    broadcast.s_addr = ulIPAddress | ~ulSubnetMask;
	
	return inet_ntoa(broadcast);
}


#define MALLOC(x) HeapAlloc(GetProcessHeap(), 0, (x))
#define FREE(x) HeapFree(GetProcessHeap(), 0, (x))
/// Note: could also use malloc() and free()

/**
* Detect all ip-addresses
*/
void WindowsSocket::getAdaptersInfo() {

// It is possible for an adapter to have multiple
// IPv4 addresses, gateways, and secondary WINS servers
// assigned to the adapter. 
//


    PIP_ADAPTER_INFO pAdapterInfo;
    PIP_ADAPTER_INFO pAdapter = NULL;
    DWORD dwRetVal = 0;

	resetFoundIPAddresses();
	
    ULONG ulOutBufLen = sizeof (IP_ADAPTER_INFO);
    pAdapterInfo = (IP_ADAPTER_INFO *) MALLOC(sizeof (IP_ADAPTER_INFO));
    if (pAdapterInfo == NULL) {
        printf("WindowsSocket::getAdaptersInfo Error allocating memory needed to call GetAdaptersinfo\n");
        return ;
    }
	// Make an initial call to GetAdaptersInfo to get
	// the necessary size into the ulOutBufLen variable
    if (GetAdaptersInfo(pAdapterInfo, &ulOutBufLen) == ERROR_BUFFER_OVERFLOW) {
        FREE(pAdapterInfo);
        pAdapterInfo = (IP_ADAPTER_INFO *) MALLOC(ulOutBufLen);
        if (pAdapterInfo == NULL) {
            printf("WindowsSocket::getAdaptersInfo Error allocating memory needed to call GetAdaptersinfo\n");
            return ;
        }
    }
    if ((dwRetVal = GetAdaptersInfo(pAdapterInfo, &ulOutBufLen)) == NO_ERROR) {
        pAdapter = pAdapterInfo;
        while (pAdapter) {
			if (strncmp(pAdapter->IpAddressList.IpAddress.String, "0.0.0.", 6) == 0) {
				pAdapter = pAdapter->Next;
				continue;
			}
			addIPAddress(pAdapter->IpAddressList.IpAddress.String, pAdapter->IpAddressList.IpMask.String);
			
            pAdapter = pAdapter->Next;		
        }
    } else {
        printf("WindowsSocket::getAdaptersInfo failed with error: %ud\n", dwRetVal);
    }
    if (pAdapterInfo)
        FREE(pAdapterInfo);
	
	//loopback is not listet-> add manually
	addIPAddress("127.0.0.1", "255.0.0.0");
	
	removeNotFoundIPAddresses();
}


#endif