/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

/* 
 * SPIt_Client handles the SPIT_Message transfer and 
 * fires eventxxx functions to the SPIT_ClientInterface (implemented by the user app). 
 * 
 * File:   SPIT_Client.cpp
 * Author: Hagi
 * 
 * Created on 7. April 2019, 08:50
 */


#include "SPIT_Client.h"
//#include <thread>
//#include <iostream>


SPITSocket* SPIT_Client::createSPITSocket() {
	SPITSocket* ptrSPITSocket = NULL;
#if defined(OS_WINDOWS)
	ptrSPITSocket = new WindowsSocket();
#endif
#if defined(OS_MAC) || defined (OS_LINUX)
	ptrSPITSocket = new UnixSocket();
#endif
	return ptrSPITSocket;
}


/**
 * Create a new SPIT_client instance, should be done once by the user app.<br>
 * @param p_SPITClientInterface the user app should implement the SPIT_ClientInterface.
 */
SPIT_Client::SPIT_Client(SPIT_ClientInterface* p_SPITClientInterface): SAD_ClientInterface() {
	m_SPITClientInterface = p_SPITClientInterface;
	m_ConnectionStatus = CONNECTION_NONE;
	m_DisconnectedByServer = false;
	m_SPITSocket = NULL;
	m_ThreadReceive = NULL;
	m_ServerID = "";
	m_ServerName = "";
	m_ServerComputerName = "";
	m_ServerStepTime = -1;
	
	m_MessageNumber_Send = 1;
	m_MessageNumber_Received = -1; //set to -1 because the first received message will set the number (1..255)
	
#if defined(OS_WINDOWS)
	WindowsSocket::init();
#endif
#if defined(OS_MAC) || defined (OS_LINUX)
	UnixSocket::init();
#endif
	m_ID = createGUID();
	m_ComputerName = getClientHostName();
	m_Watchdog = new Watchdog(this);
}

SPIT_Client::~SPIT_Client() {
	stopServerAutoDetect();

	disconnectInternal(CONNECTION_NONE);
	
	SPIT_Handler::closeHandler();
	
	
#if defined(OS_WINDOWS)
	WindowsSocket::cleanUp();
#endif
#if defined(OS_MAC) || defined (OS_LINUX)
	UnixSocket::cleanUp();
#endif
}





/** The ID of the client.<br>Every client instance should set an unique id.<br>This is done autmatically.*/
string SPIT_Client::getID() {
	return m_ID;
}
/** The name of the client's host computer*/
string SPIT_Client::getComputerName() {
	return m_ComputerName;
}
/**
 * Sets the name of the client. This should be done by the user app.
 * @param p_Name the anme of the user's app
 */
void SPIT_Client::setName(string p_Name) {
	m_Name = p_Name;
}
/** The name of the client, as set by the user's app*/
string SPIT_Client::getName() {
	return m_Name;
}

void SPIT_Client::setClientProjectName(string p_ClientProjectName) {
	m_ClientProjectName = p_ClientProjectName;
}
string SPIT_Client::getClientProjectName() {
	return m_ClientProjectName;
}

//connected Server Infos
/**If a connection is established. THis returns the SPITServer's ID*/
string SPIT_Client::getServerID() {
	return m_ServerID;
}
/**If a connection is established. THis returns the SPITServer's computer/host name*/
string SPIT_Client::getServerComputerName() {
	return m_ServerComputerName;
}
/**If a connection is established. THis returns the SPITServer's name*/
string SPIT_Client::getServerName() {
	return m_ServerName;
}
/** Expected Refreah Rate of the connectd SPITServer.<br>
 * When the server plays objects, it will do so at discrete time steps.<br>
 * Play events will take place in these time steps..
 */
int SPIT_Client::getServerStepTime() {
	return m_ServerStepTime;
}

void SPIT_Client::clientProjectClose() {
	if (m_ConnectionStatus == CONNECTION_OK) {
		SPIT_Message::CloseSPITProject* ptrSMCloseSPITProject;
		ptrSMCloseSPITProject = new SPIT_Message::CloseSPITProject();
		ptrSMCloseSPITProject->setProjectID(m_SPITClientInterface->getClientProjectID());
		if (sendSPITMessage(ptrSMCloseSPITProject) < 0) {
#if defined(DEBUGOUTPUT)
			printf("SPIT_Client could not send Client's ReportSPITProject");
#endif
		}
	}
}

void SPIT_Client::clientProjectLoaded() {
	//set confirmed = false for all SPITTypes and SPITObjects
	SPITType::setConfirmedFalse();
	SPITObject::setConfirmedFalse();
	
	if (m_ConnectionStatus == CONNECTION_OK) {
		//send ReportSPITProject to the server 
		SPIT_Message::ReportSPITProject* ptrSMReportSPITProject;
		ptrSMReportSPITProject = new SPIT_Message::ReportSPITProject();
		ptrSMReportSPITProject->setProjectID(m_SPITClientInterface->getClientProjectID());
		ptrSMReportSPITProject->setProjectName(m_SPITClientInterface->getClientProjectName());
		ptrSMReportSPITProject->setFramesPerSecond(0);
		if (sendSPITMessage(ptrSMReportSPITProject) < 0) {
#if defined(DEBUGOUTPUT)
			printf("SPIT_CLient::clientProjectLoaded could not send Client's ReportSPITProject\n");
#endif
		}
		
		//send RequestProjectTransfer
		SPIT_Message::RequestProjectTransfer* ptrSMRequestProjectTransfer;
		ptrSMRequestProjectTransfer = new SPIT_Message::RequestProjectTransfer();
		if (sendSPITMessage(ptrSMRequestProjectTransfer) < 0) {
#if defined(DEBUGOUTPUT)
			printf("SPIT_CLient::clientProjectLoaded could not send RequestProjectTransfer\n");
#endif
		}		
	}
	
	
}


/**
 * Virtual construct - means ActionAssigned for SPITTypes is is not implemented in SPITProtocol.<br>
 * This can used to setActionAssigned to all SPITObjects with this SPITType.<br>
 * This may be used in the case when the same action should be done for every SPITObject with this SPITType associated.<br>
 * This tells the server not what action is done, but it tells the server that there is an action.<br>
 * @param p_SPITTypeID the id of the SPITType
 * @param p_ActionAssigned
 */
void SPIT_Client::setActionSPITType(string p_SPITTypeID, bool p_ActionAssigned) {
	SPITType* ptrSPITType;
	SPITObject* ptrSPITObject;
	{
		unique_lock<recursive_mutex> o_LockType(*SPITType::getMutex());
		
		ptrSPITType = SPITType::getSPITType(p_SPITTypeID);
		if (ptrSPITType == NULL) return;
		if (ptrSPITType->setActionAssigned(p_ActionAssigned) == true) {
			sendSPITTypeInternal(ptrSPITType);

			//setActionAssigned to all SPITObjects with this SPITType
			unique_lock<recursive_mutex> o_LockObject(*SPITObject::getMutex());
			
			for (int i = 0; i < SPITObject::getSPITObjectCount(); i++) {
				ptrSPITObject = SPITObject::getSPITObject(i);
				if (ptrSPITObject == NULL) continue;
				if (ptrSPITObject->getSPITTypeID() == ptrSPITType->getID()) {
					if (ptrSPITObject->setActionAssignedType(p_ActionAssigned) == true) {
						sendSPITObjectInternal(ptrSPITObject);
					}
				}
			}
		}
	}
}

/**
 * This should be called if the user wants to associate/remove an action with a SPITObject.
 * @param p_SPITObjectID
 * @param p_ActionAssigned
 */
void SPIT_Client::setActionSPITObject(string p_SPITObjectID, bool p_ActionAssigned) {
	SPITObject* ptrSPITObject;	
	{
		unique_lock<recursive_mutex> o_LockObject(*SPITObject::getMutex());
		
		ptrSPITObject = SPITObject::getSPITObject(p_SPITObjectID);
		if (ptrSPITObject == NULL) return;
		if (ptrSPITObject->setActionAssigned(p_ActionAssigned) == true) {
			sendSPITObjectInternal(ptrSPITObject);
		}
	}
}
//---------- ServerAutoDetection SAD -----------

/** Starts the autodetection of SPITServers in the local network. If a connection is tried, the autodetection should be stopped before.<br>
 * If a SPITServer is detected in the locale network an eventServerDetected occurs.
 */
void SPIT_Client::startServerAutoDetect() {
	if (m_DisconnectedByServer == true) {
		//Server has previously disconnected -> do not start ServerAutodetect directly 
		//sleep a little to give the SPITServer time to clean up its stuff - 
		this_thread::sleep_for(chrono::milliseconds(1000));
	}
	SPITSocket* ptrSPITSocket;
	ptrSPITSocket = createSPITSocket();
	if (ptrSPITSocket == NULL) return;
	ptrSPITSocket->setBroadcast(true);
	SAD_Client::startAutoDetectServer(this, ptrSPITSocket);

}
/**
 * Stops the auto detection of SPIT servers.
 */
void SPIT_Client::stopServerAutoDetect() {
	SAD_Client::stopAutoDetectServer(this);
}
void SPIT_Client::eventServerDetected(string p_ID, string p_Name, string p_ComputerName, string p_IP, string p_Port, string p_Addition, string p_ProjectName, string p_ClientIP, bool p_isLoopback, bool p_IsIP4, bool p_IsIP6) {
	SAD_DetectedServer o_RefDetectedServer;
	try {
	o_RefDetectedServer = SAD_DetectedServer::getRefDetectedServer(p_ID);
	}
	catch (...) {
		return;
	}
	NotifyServerDetected* o_Runnable = new NotifyServerDetected(m_SPITClientInterface, o_RefDetectedServer);
	SPIT_Handler::addRunnable(o_Runnable);
}
void SPIT_Client::eventServerLost(string p_ID, string p_Name, string p_ComputerName, string p_IP, string p_Port, string p_Addition, string p_ProjectName, string p_ClientIP, bool p_isLoopback, bool p_IsIP4, bool p_IsIP6) {
	SAD_DetectedServer o_RefDetectedServer;
	try {
		o_RefDetectedServer = SAD_DetectedServer::getRefDetectedServer(p_ID);
	}
	catch (...) {
		return;
	}
	NotifyServerLost* o_Runnable = new NotifyServerLost(m_SPITClientInterface, o_RefDetectedServer);
	SPIT_Handler::addRunnable(o_Runnable);
}
void SPIT_Client::eventServerChanged(string p_ID, string p_Name, string p_ComputerName, string p_IP, string p_Port, string p_Addition, string p_ProjectName, string p_ClientIP, bool p_isLoopback, bool p_IsIP4, bool p_IsIP6) {
	SAD_DetectedServer o_RefDetectedServer;
	try {
	o_RefDetectedServer = SAD_DetectedServer::getRefDetectedServer(p_ID);
	}
	catch (...) {
		return;
	}
	NotifyServerChanged* o_Runnable = new NotifyServerChanged(m_SPITClientInterface, o_RefDetectedServer);
	SPIT_Handler::addRunnable(o_Runnable);
}



void SPIT_Client::fillHeader(SPIT_Message::Header* p_Header) {
	if (p_Header == NULL) return;
	p_Header->setClientID(m_ID);
	p_Header->setServerID(m_ServerID);
}

int SPIT_Client::sendSPITMessage(SPIT_Message::OPPart* p_SPITMessageOPPart) {
	if (p_SPITMessageOPPart == NULL) return -1;
	{
		unique_lock<recursive_mutex> o_LockSocket(m_MutexSPITSocket);
		
		if (m_SPITSocket == NULL || m_StopReceiveSPITMessages == true) {
			delete p_SPITMessageOPPart;
			return -1;
		}
		SPIT_Message* ptrSPIT_Message;
		const char* ptrMessageBytes;
		int intByteCount;
		int intResult;
		fillHeader(p_SPITMessageOPPart->getHeader());
		
		if (m_MessageNumber_Send == 255) m_MessageNumber_Send = 1;
		else m_MessageNumber_Send++;
		
		p_SPITMessageOPPart->getHeader()->setMessageNumber(m_MessageNumber_Send);
		
		ptrSPIT_Message = new SPIT_Message(p_SPITMessageOPPart);
		ptrMessageBytes = ptrSPIT_Message->getMessageBytes();
		intByteCount = ptrSPIT_Message->getMesageByteCount();
		if (ptrMessageBytes == NULL) {
			delete ptrSPIT_Message;
			return -1;
		}
		try {
			intResult = m_SPITSocket->sendMessage(ptrMessageBytes, intByteCount);
			if (intResult < 0) {
#if defined(DEBUGOUTPUT)
				printf("SPIT_Client::sendSPITMessage could not send message OPFlage: %d\n", ptrSPIT_Message->getOPFlag());
#endif
			}
			delete ptrSPIT_Message;
			p_SPITMessageOPPart = NULL;		
			return intResult;
		}
		catch (...) {
#if defined(DEBUGOUTPUT)
			printf("Exception SPIT_Client::sendSPITMessage OPFlag: %d\n", ptrSPIT_Message->getOPFlag());
#endif
			return -1;
		}
	}
}
bool SPIT_Client::startReceiveSPITMessages() {
	{
		unique_lock<recursive_mutex> o_LockSocket(m_MutexSPITSocket);
		if (m_SPITSocket == NULL) return false;
		if (m_SPITSocket->getBoundClientFamily() <= 0) return false;
		m_StopReceiveSPITMessages = false;
		m_ThreadReceive = new thread(&SPIT_Client::receiveSPITMessage, this);
	}
	return true;
}
void SPIT_Client::stopReceiveSPITMessages() {
	
	{
		unique_lock<recursive_mutex> o_LockSocket(m_MutexSPITSocket);
		if (m_StopReceiveSPITMessages == true) return;
		m_StopReceiveSPITMessages = true;
	}
	this_thread::sleep_for(chrono::milliseconds(100));
	if (m_ThreadReceive != NULL) {
		try {
		if (this_thread::get_id() != m_ThreadReceive->get_id()) {
			m_ThreadReceive->join();
		}
		}
		catch(...) {
#if defined(DEBUGOUTPUT)
			printf("Exception SPIT_Client::stopReceiveSPITMessages m_ThreadReceive->join()\n");
#endif
		}
		m_ThreadReceive = NULL;
		
		if (m_SPITSocket != NULL) {
			delete m_SPITSocket;
			m_SPITSocket = NULL;
		}
	}
}

void SPIT_Client::receiveSPITMessage() {
	if (m_SPITSocket == NULL) return;
	SPIT_Message* ptrSPIT_Message;
	unsigned int intActualMessageNumber;
	char buffer[MAXSPITMESSAGEBYTECOUNT];
	int intResult;
	
	
	while (m_StopReceiveSPITMessages == false) {
		{
			unique_lock<recursive_mutex> o_LockSocket(m_MutexSPITSocket);
			if (m_SPITSocket == NULL || m_StopReceiveSPITMessages == true) {
				return;
			}
		}
		intResult = m_SPITSocket->receiveMessage(m_SPITSocket->getBoundClientFamily(), buffer, MAXSPITMESSAGEBYTECOUNT, NULL);
		
		if (intResult < 0) {
			if (m_StopReceiveSPITMessages == false) {
				//To reduce the processor load, we let the thread sleep a little bit.
				this_thread::sleep_for(chrono::milliseconds(20));
//				printf("SPIT_Client::receiveSPITMessage intResult < 0\n");
				continue;
			}
			else {
				return;;
			}
		}

		ptrSPIT_Message = new SPIT_Message(buffer, intResult);
		
		intActualMessageNumber = (int)(ptrSPIT_Message->getHeader()->getMessageNumber() & 0xFF);

		checkParseSPIT_Message(ptrSPIT_Message);
	}
}

void SPIT_Client::resetMessageNumberReceived() {
	SPIT_Message* ptrOvertakingSPIT_Message;
	
	m_MessageNumber_Received = -1;
	for (auto it = m_OvertakingSPIT_Messages.begin(); it != m_OvertakingSPIT_Messages.end(); it++) {
		ptrOvertakingSPIT_Message = *it;
		delete ptrOvertakingSPIT_Message;
//		m_OvertakingSPIT_Messages.erase(it);
	}
	m_OvertakingSPIT_Messages.clear();	
}

void SPIT_Client::checkParseSPIT_Message(SPIT_Message* p_SPITMessage) {
	m_Watchdog->answerWatchdog();
	if (p_SPITMessage == NULL) {
#if defined(DEBUGOUTPUT)
		printf("SPIT_Client::parseSPIT_Message Parameter == NULL\n");
#endif
		return;
	}
	if (p_SPITMessage->getHeader() == NULL) {
#if defined(DEBUGOUTPUT)
		printf("SPIT_Client::parseSPIT_Message no Header\n");
#endif
		return;
	}
	if (p_SPITMessage->getOPPart() == NULL) {
#if defined(DEBUGOUTPUT)
		printf("SPIT_Client::parseSPIT_Message no OPPart\n");
#endif
		return;
	}
	
	int intActualMessageNumber;
	int intNextMessageNumber;
	SPIT_Message* ptrOvertakingSPIT_Message;

	intActualMessageNumber = (int)(p_SPITMessage->getHeader()->getMessageNumber() & 0xFF);

	if (intActualMessageNumber == 0) {
		parseSPIT_Message(p_SPITMessage);
		return;
	}
	if (m_MessageNumber_Received < 0) {
		m_MessageNumber_Received = intActualMessageNumber;
		parseSPIT_Message(p_SPITMessage);
		return;
	}
	
	intNextMessageNumber = m_MessageNumber_Received +1;
	if (intNextMessageNumber > 255 || intNextMessageNumber < 1) intNextMessageNumber = 1;
		
	if (intActualMessageNumber == intNextMessageNumber) {
		m_MessageNumber_Received = intActualMessageNumber;
		parseSPIT_Message(p_SPITMessage);
			
		//check overtaking messages
		auto it = m_OvertakingSPIT_Messages.begin();
		while (m_OvertakingSPIT_Messages.size() > 0) {
				
			intNextMessageNumber = m_MessageNumber_Received +1;
			if (intNextMessageNumber > 255 || intNextMessageNumber < 1) intNextMessageNumber = 1;
	
			ptrOvertakingSPIT_Message = *it;
			intActualMessageNumber = (int)(ptrOvertakingSPIT_Message->getHeader()->getMessageNumber() & 0xFF);
				
			if (intActualMessageNumber == intNextMessageNumber) {
				m_MessageNumber_Received = intActualMessageNumber;
				parseSPIT_Message(ptrOvertakingSPIT_Message);
				m_OvertakingSPIT_Messages.erase(it);
					
				it = m_OvertakingSPIT_Messages.begin();
			}
			else {
				break;
			}
		}
			
	}
	else {
		//p_SPITMessage has overtaken another message
		addOvertakingSPIT_Message(p_SPITMessage);			
	}
	
}

int SPIT_Client::s_MAX_OVERTAKING_SPIT_MESSAGES = 10;

void SPIT_Client::addOvertakingSPIT_Message(SPIT_Message* p_SPIT_Message) {
	if (p_SPIT_Message == nullptr) return;
	int intMessageNumber;
	SPIT_Message* ptrOvertakingSPIT_Message;
	int intOvertakingMessageNumber;
	
	if (m_OvertakingSPIT_Messages.size() > s_MAX_OVERTAKING_SPIT_MESSAGES) {
		printf("SPIT_Client receiveing: perhaps message lost !!!!!");
	}
	
	intMessageNumber = (int)(p_SPIT_Message->getHeader()->getMessageNumber() & 0xFF);
	for (auto it = m_OvertakingSPIT_Messages.begin(); it != m_OvertakingSPIT_Messages.end(); it++) {
		ptrOvertakingSPIT_Message = *it;
		intOvertakingMessageNumber = (int)(ptrOvertakingSPIT_Message->getHeader()->getMessageNumber() & 0xFF);
		if (intMessageNumber < intOvertakingMessageNumber) {
			m_OvertakingSPIT_Messages.insert(it, p_SPIT_Message);
			return;
		}
	}
	m_OvertakingSPIT_Messages.push_back(p_SPIT_Message);
}

void SPIT_Client::parseSPIT_Message(SPIT_Message* p_SPITMessage) {
	m_Watchdog->answerWatchdog();
	if (p_SPITMessage == NULL) {
#if defined(DEBUGOUTPUT)
		printf("SPIT_Client::parseSPIT_Message Parameter == NULL\n");
#endif
		return;
	}
	if (p_SPITMessage->getHeader() == NULL) {
#if defined(DEBUGOUTPUT)
		printf("SPIT_Client::parseSPIT_Message no Header\n");
#endif
		return;
	}
	if (p_SPITMessage->getOPPart() == NULL) {
#if defined(DEBUGOUTPUT)
		printf("SPIT_Client::parseSPIT_Message no OPPart\n");
#endif
		return;
	}
	//----- Connection -----
	if (p_SPITMessage->getOPFlag() == SPIT_Message::ConnectAnswer::OPFLAG) {
		SPIT_Message::ConnectAnswer* o_ConnectAnswer = (SPIT_Message::ConnectAnswer*) p_SPITMessage->getOPPart();
		parseConnectAnswer(o_ConnectAnswer);
	}
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::Disconnect::OPFLAG) {
		SPIT_Message::Disconnect* o_Disconnect = (SPIT_Message::Disconnect*) p_SPITMessage->getOPPart();
		parseDisconnect(o_Disconnect);
	}
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::ReportSPITProject::OPFLAG) {
		SPIT_Message::ReportSPITProject* o_ReportSPITProject = (SPIT_Message::ReportSPITProject*) p_SPITMessage->getOPPart();
		parseReportSPITProject(o_ReportSPITProject);
	}
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::Watchdog::OPFLAG) {
		SPIT_Message::Watchdog* o_Watchdog = (SPIT_Message::Watchdog*) p_SPITMessage->getOPPart();
		parseWatchdog(o_Watchdog);
	}
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::WatchdogAnswer::OPFLAG) {
		SPIT_Message::WatchdogAnswer* o_WatchdogAnswer = (SPIT_Message::WatchdogAnswer*) p_SPITMessage->getOPPart();
		parseWatchdogAnswer(o_WatchdogAnswer);
	}
	//----- SPIT Project -----
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::StartProjectTransfer::OPFLAG) {
		SPIT_Message::StartProjectTransfer* o_StartProjectTransfer = (SPIT_Message::StartProjectTransfer*) p_SPITMessage->getOPPart();
		parseStartProjectTransfer(o_StartProjectTransfer);
	}
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::CloseSPITProject::OPFLAG) {
		SPIT_Message::CloseSPITProject* o_CloseSPITProjectr = (SPIT_Message::CloseSPITProject*) p_SPITMessage->getOPPart();
		parseCloseSPITProject(o_CloseSPITProjectr);
	}
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::RequestProjectTransfer::OPFLAG) {
		SPIT_Message::RequestProjectTransfer* o_RequestProjectTransfer = (SPIT_Message::RequestProjectTransfer*) p_SPITMessage->getOPPart();
		parseRequestProjectTransfer(o_RequestProjectTransfer);
	}
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::StartProjectTransfer::OPFLAG) {
		SPIT_Message::StartProjectTransfer* o_StartProjectTransfer = (SPIT_Message::StartProjectTransfer*) p_SPITMessage->getOPPart();
		parseStartProjectTransfer(o_StartProjectTransfer);
	}
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::EndProjectTransfer::OPFLAG) {
		SPIT_Message::EndProjectTransfer* o_EndProjectTransfer = (SPIT_Message::EndProjectTransfer*) p_SPITMessage->getOPPart();
		parseEndProjectTransfer(o_EndProjectTransfer);
	}
	//----- SPIT Types ------
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::ReportSPITType::OPFLAG) {
		SPIT_Message::ReportSPITType* o_ReportSPITType = (SPIT_Message::ReportSPITType*) p_SPITMessage->getOPPart();
		parseReportSPITType(o_ReportSPITType);
	}
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::RemoveSPITType::OPFLAG) {
		SPIT_Message::RemoveSPITType* o_RemoveSPITType = (SPIT_Message::RemoveSPITType*) p_SPITMessage->getOPPart();
		parseRemoveSPITType(o_RemoveSPITType);
	}
	//----- SPIT Objects ------
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::ReportSPITObject::OPFLAG) {
		SPIT_Message::ReportSPITObject* o_ReportSPITObject = (SPIT_Message::ReportSPITObject*) p_SPITMessage->getOPPart();
		parseReportSPITObject(o_ReportSPITObject);
	}
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::RemoveSPITObject::OPFLAG) {
		SPIT_Message::RemoveSPITObject* o_RemoveSPITObject = (SPIT_Message::RemoveSPITObject*) p_SPITMessage->getOPPart();
		parseRemoveSPITObject(o_RemoveSPITObject);
	}
	//----- Play SPIT Objects ------
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::PlaySPITStart::OPFLAG) {
		SPIT_Message::PlaySPITStart* o_PlaySPITStart = (SPIT_Message::PlaySPITStart*) p_SPITMessage->getOPPart();
		parsePlaySPITStart(o_PlaySPITStart);
	}
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::PlaySPITObject::OPFLAG) {
		SPIT_Message::PlaySPITObject* o_PlaySPITObject = (SPIT_Message::PlaySPITObject*) p_SPITMessage->getOPPart();
		parsePlaySPITObject(o_PlaySPITObject);
	}
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::PlaySPITEnd::OPFLAG) {
		SPIT_Message::PlaySPITEnd* o_PlaySPITEnd = (SPIT_Message::PlaySPITEnd*) p_SPITMessage->getOPPart();
		parsePlaySPITEnd(o_PlaySPITEnd);
	}
	else if (p_SPITMessage->getOPFlag() == SPIT_Message::ReportPlayStatus::OPFLAG) {
		SPIT_Message::ReportPlayStatus* o_ReportPlayStatus = (SPIT_Message::ReportPlayStatus*) p_SPITMessage->getOPPart();
		parseReportPlayStatus(o_ReportPlayStatus);
	}
	else {
#if defined(DEBUGOUTPUT)
		printf("SPIT_Client::parseSPIT_Message: Unknown Message\n");
#endif
	}
	
	delete p_SPITMessage;
}


//----- Connection ------
/**
 * Tries to connect to a SPITServer.<br>
 * @param p_ServerIP the ip address of the SPIT server
 * @param p_ServerPort the port number of the SPIT server
 * @param p_ClientIP the local IP address of the client
 * @param p_ClientPort the port number of the Client, if "0" the OS uses any possible port number 
 * @param p_LoginName login user name (liveSHOW does not use this -> mayby ""
 * @param p_LoginPassword login user password (liveSHOW does not use this -> mayby ""
 * @return false = an error has occured, true = the connect was sent to the server
 */
bool SPIT_Client::connect(string p_ServerIP, string p_ServerPort, string p_ClientIP, string p_ClientPort, string p_LoginName, string p_LoginPassword) {
	int intResult;
	disconnect();
		
	{
		unique_lock<recursive_mutex> o_LockSocket(m_MutexSPITSocket);
		m_SPITSocket = createSPITSocket();
		m_SPITSocket->setConnectionParams(p_ServerIP, p_ServerPort, p_ClientIP, p_ClientPort);
		intResult = m_SPITSocket->bindSocket(p_ServerIP, p_ClientIP, p_ClientPort);
		if (intResult < 0) {
			m_StopReceiveSPITMessages = true;
			delete m_SPITSocket;
			m_SPITSocket = NULL;
			return false;
		}
	}
	SPIT_Message::Connect* ptrSMConnect;
	ptrSMConnect = new SPIT_Message::Connect();
	ptrSMConnect->setClientComputerName(m_ComputerName);
	ptrSMConnect->setClientName(m_Name);
	ptrSMConnect->setLoginName(p_LoginName);
	ptrSMConnect->setLoginPassword(p_LoginPassword);
	ptrSMConnect->setServerIP(p_ServerIP);
	if (p_ServerPort.length() <= 0) {
		p_ServerPort = "0";
	}
	int intServerPort;
	intServerPort = stoi(p_ServerPort);
	ptrSMConnect->setServerPort(intServerPort);
	
	if (startReceiveSPITMessages() == false) {
		{
			unique_lock<recursive_mutex> o_LockSocket(m_MutexSPITSocket);
			m_StopReceiveSPITMessages = true;
			delete m_SPITSocket;
			m_SPITSocket = NULL;
			return false;
		}
	}
	
	resetMessageNumberReceived();
	
	int intOldConnectionStatus = m_ConnectionStatus;
	
	m_ConnectionStatus = CONNECTION_TRYCONNECTION;
	NotifyConnectionStatus* o_Runnable = new NotifyConnectionStatus(m_SPITClientInterface, m_ServerID, m_ServerComputerName, m_ServerName, m_ServerStepTime, 
				m_ConnectionStatus, intOldConnectionStatus);
	SPIT_Handler::addRunnable(o_Runnable);
	
	//sleep a little to make sure that the reception of the messages is started 
	this_thread::sleep_for(chrono::milliseconds(500));
	intResult = sendSPITMessage(ptrSMConnect);
	if (intResult < 0) {
		m_ConnectionStatus = CONNECTION_NONE;
		{
			unique_lock<recursive_mutex> o_LockSocket(m_MutexSPITSocket);
			m_StopReceiveSPITMessages = true;
			delete m_SPITSocket;
			m_SPITSocket = NULL;
			return false;
		}
	}
	//WAITING FOR CONNECTION ANSWER
	//If the Server is closed in the meantime no answer will return
	//-> startWatchdog
	m_Watchdog->startWatchdog();
	return true;
}

/**
 * Terminates the connection to the SPITServer.
 */
void SPIT_Client::disconnect() {
	disconnectInternal(CONNECTION_NONE);
}
void SPIT_Client::disconnectInternal(int p_Cause) {
	int intOldConnectionStatus = m_ConnectionStatus;
	SPIT_Message::Disconnect* ptrSMDisconnect;
	int intResult;
	
	m_Watchdog->stopWatchdog();

	m_ConnectionStatus = p_Cause;
	switch (m_ConnectionStatus) {
		case CONNECTION_LOST:
			break;
		case CONNECTION_NONE:
			ptrSMDisconnect = new SPIT_Message::Disconnect();
			intResult = sendSPITMessage(ptrSMDisconnect);
			if (intResult < 0) {
			}
			break;
		case CONNECTION_NONESERVER:
			break;
	}
	stopReceiveSPITMessages();
	
//	if (intOldConnectionStatus != CONNECTION_NONE && intOldConnectionStatus != CONNECTION_NONESERVER) {
		NotifyConnectionStatus* o_Runnable = new NotifyConnectionStatus(m_SPITClientInterface, m_ServerID, m_ServerComputerName, m_ServerName, m_ServerStepTime, 
				m_ConnectionStatus, intOldConnectionStatus);
		SPIT_Handler::addRunnable(o_Runnable);
//	}
}

void SPIT_Client::parseConnectAnswer(SPIT_Message::ConnectAnswer* p_ConnectAnswer) {
	int intOldConnectionStatus;
	intOldConnectionStatus = m_ConnectionStatus;

	if (p_ConnectAnswer->getClientID() != m_ID) return;

	m_ServerID = p_ConnectAnswer->getServerID();
	m_ServerComputerName = p_ConnectAnswer->getServerComputerName();
	m_ServerName = p_ConnectAnswer->getServerName();
	m_ServerStepTime = p_ConnectAnswer->getStepTime();
	
	if (p_ConnectAnswer->getConnected() == false) {
		disconnectInternal(CONNECTION_NONESERVER);
	}
	else {
		p_ConnectAnswer->getStepTime();
		m_ConnectionStatus = CONNECTION_OK;
		m_DisconnectedByServer = false;
		m_Watchdog->startWatchdog();
		NotifyConnectionStatus* o_Runnable = new NotifyConnectionStatus(m_SPITClientInterface, m_ServerID, m_ServerComputerName, m_ServerName, m_ServerStepTime, 
				m_ConnectionStatus, intOldConnectionStatus);
		SPIT_Handler::addRunnable(o_Runnable);
	}
}

/**
 * The SPIT server was closed
 * @param p_Disconnect
 */
void SPIT_Client::parseDisconnect(SPIT_Message::Disconnect* p_Disconnect) {
	int intCauseDisconnect = CONNECTION_NONESERVER;
	/*
	function<void()> o_Callable = bind(&SPIT_Client::disconnectInternal, this, intCauseDisconnect);
	SPIT_Handler::addCallable(o_Callable);
	*/
	m_DisconnectedByServer = true;
	SPIT_Client::disconnectInternal(intCauseDisconnect);
}


//----- Watchdog ------
/**
 * The SPIT server sends a watchdog message at regular intervals. The client responds immediately with WatchdogAnswer<br>
 * Conversely, the client also sends a watchdog message at regular intervals and waits for a WatchdogAnswer message.<br>
 * This is used to check if the other side is still present.<br>  
 * @param p_Watchdog
 */
void SPIT_Client::parseWatchdog(SPIT_Message::Watchdog* p_Watchdog) {
//	printf("parse Watchdog \n");
	SPIT_Message::WatchdogAnswer* o_WatchdogAnswer;
	o_WatchdogAnswer = new SPIT_Message::WatchdogAnswer();
	int intResult;
	intResult = sendSPITMessage(o_WatchdogAnswer);
	if (intResult < 0) {
	}
}
/**
 * The SPIT server sends a watchdog message at regular intervals. The client responds immediately with WatchdogAnswer<br>
 * Conversely, the client also sends a watchdog message at regular intervals and waits for a WatchdogAnswer message.<br>
 * This is used to check if the other side is still present.<br>  
 * @param p_Watchdog
 */
void SPIT_Client::parseWatchdogAnswer(SPIT_Message::WatchdogAnswer* p_WatchdogAnswer) {
	m_Watchdog->answerWatchdog(); //is also done by parseSPIT_Message
}
void SPIT_Client::sendWatchdog() {
	int intResult;
		SPIT_Message::Watchdog* ptrSMWatchdog;
		ptrSMWatchdog = new SPIT_Message::Watchdog();
		intResult = sendSPITMessage(ptrSMWatchdog);
		if (intResult < 0) {
		}
}
void SPIT_Client::eventWatchdogLost() {
	if (m_ConnectionStatus == CONNECTION_TRYCONNECTION) {
		disconnectInternal(CONNECTION_NONESERVER);
	}
	else {
		disconnectInternal(CONNECTION_LOST);
	}
}

//----- SPIT Project ------
void SPIT_Client::parseReportSPITProject(SPIT_Message::ReportSPITProject* p_ReportSPITProject) {
//	printf("parse ReportSPITProject \n");
	
	if (p_ReportSPITProject == NULL) return;
	
	m_ServerProjectID = p_ReportSPITProject->getProjectID();
	m_ServerProjectName = p_ReportSPITProject->getProjectName();
	m_ServerFramesPerSecond = p_ReportSPITProject->getFramesPerSecond();
	/*
	function<void()> o_Callable = bind(&SPIT_Client::notifyReportProject, this);
	SPIT_Handler::addCallable(o_Callable);
	*/
	
	NotifyReportProject* o_Runnable = new NotifyReportProject(m_SPITClientInterface, m_ServerProjectID, m_ServerProjectName, m_ServerFramesPerSecond);
	SPIT_Handler::addRunnable(o_Runnable);
	
	//Set all SPITTypes and SPITObjects confirmed = false
	SPITType::setConfirmedFalse();
	SPITObject::setConfirmedFalse();
	
	
	//Send RequestProjectTransfer
	SPIT_Message::RequestProjectTransfer* ptrSMRequestProjectTransfer;
	ptrSMRequestProjectTransfer = new SPIT_Message::RequestProjectTransfer();
	if (sendSPITMessage(ptrSMRequestProjectTransfer) < 0) {
		printf("SPIT_Client::parseReportSPITProject could not send RequestProjectTransfer\n");
	}
/*
	//Don'T send the ReportSPITProject here it should be sended after parseEndProjectTransfer
 
	//Send client's ReportSPITProject
	SPIT_Message::ReportSPITProject* ptrSMReportSPITProject;
	ptrSMReportSPITProject = new SPIT_Message::ReportSPITProject();
	ptrSMReportSPITProject->setProjectID(m_SPITClientInterface->getClientProjectID());
	ptrSMReportSPITProject->setProjectName(m_SPITClientInterface->getClientProjectName());
	ptrSMReportSPITProject->setFramesPerSecond(0);
	if (sendSPITMessage(ptrSMReportSPITProject) < 0) {
		printf("SPIT_Client::parseReportSPITProject could not send Client's ReportSPITProject");
	}
*/
	
}
void SPIT_Client::parseCloseSPITProject(SPIT_Message::CloseSPITProject* p_CloseSPITProject) {
//	printf("parse CloseSPITProject \n");
	if (p_CloseSPITProject == NULL) return;
	string stringProjectID;
//	stringProjectID = p_CloseSPITProject->getProjectID();
	NotifyProjectClosed* o_Runnable = new NotifyProjectClosed(m_SPITClientInterface, stringProjectID);
	SPIT_Handler::addRunnable(o_Runnable);
	SPITPlayer::clearPlayers();
}


string SPIT_Client::getServerProjectID() {
	return m_ServerProjectID;
}
string SPIT_Client::getServerProjectName() {
	return m_ServerProjectName;
}
int SPIT_Client::getServerFramesPerSecond() {
	return m_ServerFramesPerSecond;
}

void SPIT_Client::parseRequestProjectTransfer(SPIT_Message::RequestProjectTransfer* p_RequestProjectTransfer) {
//	printf("parse RequestProjectTransfer \n");
	thread* o_ThreadSend;
	o_ThreadSend = new thread(&this->sendProject, this);
/*
	int intResult;
	SPIT_Message::StartProjectTransfer* ptrSMStartProjectTransfer;
	SPIT_Message::EndProjectTransfer* ptrSMEndProjectTransfer;
	int intCount;
	SPITType* ptrSPITType;
	SPITObject* ptrSPITObject;
	
	
	//StartProjectTransfer
	NotifyStartProjectTransfer_Client* o_RunnableStart = new NotifyStartProjectTransfer_Client(m_SPITClientInterface);
	SPIT_Handler::addRunnable(o_RunnableStart);	

	ptrSMStartProjectTransfer = new SPIT_Message::StartProjectTransfer();
	intResult = sendSPITMessage(ptrSMStartProjectTransfer);
	if (intResult < 0) {
		return;
	}
	m_Watchdog->answerWatchdog();
	{
		//send all listes SPITTypes
		unique_lock<recursive_mutex> o_LockType(*SPITType::getMutex());
		intCount = SPITType::getSPITTypeCount();
		for (int i = 0; i < intCount; i++) {
			ptrSPITType = SPITType::getSPITType(i);
			if (ptrSPITType == NULL) continue;
			intResult = sendSPITTypeInternal(ptrSPITType);
			if (intResult < 0) return;
			m_Watchdog->answerWatchdog();
			this_thread::sleep_for(chrono::milliseconds(7));
		}
	}
	
	{
		unique_lock<recursive_mutex> o_LockObject(*SPITObject::getMutex());
		intCount = SPITObject::getSPITObjectCount();
		for (int i = 0; i < intCount; i++) {
			ptrSPITObject = SPITObject::getSPITObject(i);
			if (ptrSPITObject == NULL) continue;
			intResult = sendSPITObjectInternal(ptrSPITObject);
			if (intResult < 0) return;
			m_Watchdog->answerWatchdog();
			this_thread::sleep_for(chrono::milliseconds(7));
		}
	}

	//EndProjectTransfer
	ptrSMEndProjectTransfer = new SPIT_Message::EndProjectTransfer();
	intResult = sendSPITMessage(ptrSMEndProjectTransfer);
	if (intResult < 0) {
		return;
	}
	m_Watchdog->answerWatchdog();
	NotifyEndProjectTransfer_Client* o_RunnableEnd = new NotifyEndProjectTransfer_Client(m_SPITClientInterface);
	SPIT_Handler::addRunnable(o_RunnableEnd);	
	
	//Send act
*/	
}


void SPIT_Client::sendProject(SPIT_Client* p_PtrSPIT_Client) {
	int intResult;
	SPIT_Message::StartProjectTransfer* ptrSMStartProjectTransfer;
	SPIT_Message::EndProjectTransfer* ptrSMEndProjectTransfer;
	int intCount;
	SPITType* ptrSPITType;
	SPITObject* ptrSPITObject;
	
	//StartProjectTransfer
	NotifyStartProjectTransfer_Client* o_RunnableStart = new NotifyStartProjectTransfer_Client(p_PtrSPIT_Client->getSPITClientInterface());
	SPIT_Handler::addRunnable(o_RunnableStart);	

	ptrSMStartProjectTransfer = new SPIT_Message::StartProjectTransfer();
	intResult = p_PtrSPIT_Client->sendSPITMessage(ptrSMStartProjectTransfer);
	if (intResult < 0) {
		return;
	}

//	m_Watchdog->answerWatchdog();
//	{
		//send all listes SPITTypes
//		unique_lock<recursive_mutex> o_LockType(*SPITType::getMutex());
		intCount = SPITType::getSPITTypeCount();
		for (int i = 0; i < intCount; i++) {
			ptrSPITType = SPITType::getSPITType(i);
			if (ptrSPITType == NULL) continue;
			if (ptrSPITType->getCreatedByClient() == false) continue;
			intResult = p_PtrSPIT_Client->sendSPITTypeInternal(ptrSPITType);
			if (intResult < 0) return;
//			m_Watchdog->answerWatchdog();
			this_thread::sleep_for(chrono::milliseconds(7));
		}
//	}
	
//	{
//		unique_lock<recursive_mutex> o_LockObject(*SPITObject::getMutex());
		intCount = SPITObject::getSPITObjectCount();
		for (int i = 0; i < intCount; i++) {
			ptrSPITObject = SPITObject::getSPITObject(i);
			if (ptrSPITObject == NULL) continue;
			if (ptrSPITObject->getCreatedByClient() == false) continue;
			intResult = p_PtrSPIT_Client->sendSPITObjectInternal(ptrSPITObject);
			if (intResult < 0) return;
//			m_Watchdog->answerWatchdog();
			this_thread::sleep_for(chrono::milliseconds(7));
		}
//	}

	//EndProjectTransfer
	ptrSMEndProjectTransfer = new SPIT_Message::EndProjectTransfer();
	intResult = p_PtrSPIT_Client->sendSPITMessage(ptrSMEndProjectTransfer);
	if (intResult < 0) {
		return;
	}
//	m_Watchdog->answerWatchdog();
	NotifyEndProjectTransfer_Client* o_RunnableEnd = new NotifyEndProjectTransfer_Client(p_PtrSPIT_Client->getSPITClientInterface());
	SPIT_Handler::addRunnable(o_RunnableEnd);	
	
	//Send act
	
	
}

void SPIT_Client::parseStartProjectTransfer(SPIT_Message::StartProjectTransfer* p_StartProjectTransfer) {
//printf("parse StartProjectTransfer \n");
	NotifyStartProjectTransfer_Server* o_Runnable = new NotifyStartProjectTransfer_Server(m_SPITClientInterface);
	SPIT_Handler::addRunnable(o_Runnable);	
}
void SPIT_Client::parseEndProjectTransfer(SPIT_Message::EndProjectTransfer* p_EndProjectTransfer) {
	//close all unconfirmed objects not created by this client
	SPITObject::clearUnconfirmedSPITObjects();
	
//printf("parse EndProjectTransfer \n");
	NotifyEndProjectTransfer_Server* o_Runnable = new NotifyEndProjectTransfer_Server(m_SPITClientInterface);
	SPIT_Handler::addRunnable(o_Runnable);	
	
	//Send SPIT_Message::ReportSPITProject after the m_SPITClientInterface is notfied of EndProjectTransfer 
	SendReportProject_Client* o_SendReportProject_Client = new SendReportProject_Client(this, m_SPITClientInterface);
	SPIT_Handler::addRunnable(o_SendReportProject_Client);
}


//----- SPIT Types ------
void SPIT_Client::parseReportSPITType(SPIT_Message::ReportSPITType* p_ReportSPITType) {
//	printf("parse ReportSPITType ID %s Name %s \n",p_ReportSPITType->getID(), p_ReportSPITType->getName());
	SPITType* ptrSPITType;
	bool boolNew = false;
	bool boolChanged = false;
	
	{
		unique_lock<recursive_mutex> o_LockSPITType(*SPITType::getMutex());
		
		if (p_ReportSPITType->getServerRemote() == false) {
			ptrSPITType = SPITType::getSPITType(p_ReportSPITType->getID());
			if (ptrSPITType == NULL) {
				if (m_SPITClientInterface != NULL) {
					//The user's app my veto the SPITType
					if (m_SPITClientInterface->checkSPITTypeVeto(p_ReportSPITType) == false) {
						return;
					}
				}
				ptrSPITType = new SPITType();
				boolNew = true;
			}
		}
		else {
			ptrSPITType = SPITType::getSPITTypeRemote(p_ReportSPITType->getID());
			if (ptrSPITType == NULL) {
				ptrSPITType = new SPITType();
				boolNew = true;
			}
		}
		try {
			boolChanged = ptrSPITType->setMessageOP(p_ReportSPITType);
		}
		catch (...) {
			sendSPITTypeInternal(ptrSPITType);
			return;
		}
		
		
		//add or change SPITType
		if (ptrSPITType->getServerRemote() == false) {		
			SPITType::addSPITType(ptrSPITType);
		}
		else {		
			SPITType::addSPITTypeRemote(ptrSPITType);
		}
		
		NotifyReportSPITType* o_Runnable = new NotifyReportSPITType(ptrSPITType->getServerRemote(), boolNew, m_SPITClientInterface, *ptrSPITType);
		SPIT_Handler::addRunnable(o_Runnable);
		
		if (ptrSPITType != NULL) {
			if (ptrSPITType->setConfirmed(true) == true || boolChanged == true) {
				sendSPITTypeInternal(ptrSPITType);
			}
		}	
	}
}
void SPIT_Client::parseRemoveSPITType(SPIT_Message::RemoveSPITType* p_RemoveSPITType) {
//	printf("parse RemoveSPITType %s\n", p_RemoveSPITType->getID());
	if (p_RemoveSPITType == NULL) return;
	string stringSPITTypeID;
	bool boolRemote = false;
	SPITType* ptrSPITType;
	stringSPITTypeID = p_RemoveSPITType->getID();
	
	{
		unique_lock<recursive_mutex> o_LockSPITType(*SPITType::getMutex());
		
		ptrSPITType = SPITType::getSPITTypeRemote(stringSPITTypeID);
		if (ptrSPITType != NULL) {
			boolRemote = true;
		}
		else {
			ptrSPITType = SPITType::getSPITType(stringSPITTypeID);
			if (ptrSPITType != NULL) {
				boolRemote = false;
			}
			else {
				//No SPITType with stringSPITTypeID is listet
				return;
			}
		}
		
		if (ptrSPITType->getCreatedByClient() && ptrSPITType->getFixed()) {
			//the SPITyte is created by client himself and it is fixed -> don't remove the SPITType!!
			//Send the SPITTyp back to the server
			sendSPITTypeInternal(ptrSPITType);
			return;
		}

		NotifyRemoveSPITType* o_Runnable = new NotifyRemoveSPITType(boolRemote, m_SPITClientInterface, *ptrSPITType);
		SPIT_Handler::addRunnable(o_Runnable);

		if (boolRemote == false) {
			SPITType::removeSPITType(ptrSPITType);
		}
		else {
			SPITType::removeSPITTypeRemote(ptrSPITType);
		}
		delete ptrSPITType;
		
	}
}


/**
 * If a client has changed a SPITType, it should send it to the server to transmit the changes.<br>
 * If the SPITType is a fixed one and not created by the client himself, then an eventSPITType can be sent from the server, which cancels the changes.<br>
 * @param p_SPITType the pointer to the changed SPITType 
 * @return 0 = is sended, -1 = is not sended
 */
int SPIT_Client::sendSPITTypeInternal(SPITType* p_SPITType) {
	if (p_SPITType == NULL) return -1;
	SPIT_Message::ReportSPITType* ptrSMReportSPITType;
	ptrSMReportSPITType = new SPIT_Message::ReportSPITType();
	p_SPITType->fillMessageOP(ptrSMReportSPITType);
	return sendSPITMessage(ptrSMReportSPITType);
}

/**
 * Change an existing SPIT_Type.<br>
 * @param p_SPITType the SPITType vont
 * @return true = changed, false = not changed maybe p_SPITType is NULL or the SPITType doesn't exists
 */
bool SPIT_Client::changeSPITType(SPITType* p_SPITType) {
	if (p_SPITType == NULL) return false;
	SPITType* ptrSPITType;
	{
		unique_lock<recursive_mutex> o_LockType(*SPITType::getMutex());
		
		ptrSPITType = SPITType::getSPITType(p_SPITType->getID());
		if (ptrSPITType == NULL) return false;
		
		ptrSPITType->setDefaultDelay(p_SPITType->getDefaultDelay());
		ptrSPITType->setDefaultDuration(p_SPITType->getDefaultDuration());
		ptrSPITType->setFixed(p_SPITType->getFixed());
		ptrSPITType->setGroupName(p_SPITType->getGroupName());
		ptrSPITType->setName(p_SPITType->getName());
		ptrSPITType->setProjectIndepended(p_SPITType->getProjectIndepended());
		
		NotifyReportSPITType* o_Runnable = new NotifyReportSPITType(false, false, m_SPITClientInterface, *ptrSPITType);
		SPIT_Handler::addRunnable(o_Runnable);
		
		sendSPITTypeInternal(ptrSPITType);
	}
	return true;
}
/**
 * If a client has removed a SPITType where id == p_SPITTypeID, it should send it to the server.<br>
 * The corresponding SPITType if listet in SPITType will be deleted.<br>
 * An eventRemoveSPITType will occur.<br>
 * If a connection to a SPITServer is established, the SPITServer may dicard the deletion an an eventSPITType will occur<br>
 * -> the SPITType will be listet again.<br> 
 * @param p_SPITType the id of the SPITType ro remove 
 * @return 0 = is sended, -1 = is not sended
 */
int SPIT_Client::removeSPITType(string p_SPITTypeID) {
	if (p_SPITTypeID.length() <= 0) return -1;
	SPITType* ptrSPITType;
	SPIT_Message::RemoveSPITType* ptrSMRemoveSPITType;
	{
		unique_lock<recursive_mutex> o_LockType(*SPITType::getMutex());
		
		ptrSPITType = SPITType::getSPITType(p_SPITTypeID);
		if (ptrSPITType == NULL) return -1;
		ptrSMRemoveSPITType = new SPIT_Message::RemoveSPITType();
		ptrSPITType->fillMessageOP(ptrSMRemoveSPITType);
		NotifyRemoveSPITType* o_Runnable = new NotifyRemoveSPITType(false, m_SPITClientInterface, *ptrSPITType);
		SPIT_Handler::addRunnable(o_Runnable);
		SPITType::removeSPITType(ptrSPITType);	
		delete ptrSPITType;
	}
	return sendSPITMessage(ptrSMRemoveSPITType);
}


SPITType* SPIT_Client::createSPITTypeFixed(string p_ID, string p_GroupName, string p_Name, long long p_DefaultDelay, long long p_DefaultDuration) {
	return createSPITType(p_ID, p_GroupName, p_Name, p_DefaultDelay, p_DefaultDuration, true, true);
}

SPITType* SPIT_Client::createSPITType(string p_ID, string p_GroupName, string p_Name, long long p_DefaultDelay, long long p_DefaultDuration, bool p_FIxed, bool p_ProjectIndepended) {
	SPITType* ptrSPITType;
	bool boolNew;
	{
		unique_lock<recursive_mutex> o_LockType(*SPITType::getMutex());
		
		ptrSPITType = SPITType::getSPITType(p_ID);
		if (ptrSPITType == NULL) {
			ptrSPITType = new SPITType(p_ID, p_GroupName, p_Name, p_FIxed, p_ProjectIndepended, p_DefaultDelay, p_DefaultDuration);
			ptrSPITType->setCreatedByClient(true);
			boolNew = true;
		}
		else {
			ptrSPITType->setID(p_ID);
			ptrSPITType->setGroupName(p_GroupName);
			ptrSPITType->setName(p_Name);
			ptrSPITType->setFixed(p_FIxed);
			ptrSPITType->setProjectIndepended(p_ProjectIndepended);
			ptrSPITType->setDefaultDelay(p_DefaultDelay);
			ptrSPITType->setDefaultDuration(p_DefaultDuration);
			boolNew = false;
		}
	}
	SPITType::addSPITType(ptrSPITType);
	NotifyReportSPITType* o_Runnable = new NotifyReportSPITType(false, boolNew, m_SPITClientInterface, *ptrSPITType);
	SPIT_Handler::addRunnable(o_Runnable);
	sendSPITTypeInternal(ptrSPITType);
	return ptrSPITType;
} 
	




//----- SPIT Objects ------

void SPIT_Client::parseReportSPITObject(SPIT_Message::ReportSPITObject* p_ReportSPITObject) {
	
//printf("parse ReportSPITObject %s\n", p_ReportSPITObject->getID());
	if (p_ReportSPITObject == NULL) return;
	SPITObject* ptrSPITObject;
	SPITType* ptrSPITType;
	bool boolNew = false;
	bool boolRemote = false;	
	bool boolChanged = false;
	{
		unique_lock<recursive_mutex> o_LockSPITObject(*SPITObject::getMutex());
		unique_lock<recursive_mutex> o_LockSPITType(*SPITType::getMutex());
		
		if (p_ReportSPITObject->getServerRemote() == false) {
			boolRemote = false;
			ptrSPITObject = SPITObject::getSPITObject(p_ReportSPITObject->getID());
			ptrSPITType = SPITType::getSPITType(p_ReportSPITObject->getSPITTypeID());
			
			if (ptrSPITType == NULL) {
				//No associated SPITType found
				if (ptrSPITObject != NULL) {
					//SpitObject exists but has no corresponding SPITType -> remove the SPITObject without sending a message
					//Maybe the SPITObject has changed the SPITTypeID to a SPITTypeID of another board
					SPITObject::removeSPITObject(ptrSPITObject);
					delete ptrSPITObject;
				}
				return;
			}
			if (ptrSPITObject == NULL) {
				if (m_SPITClientInterface != NULL) {
					//The user's app my veto the SPITType
					if (m_SPITClientInterface->checkSPITObjectVeto(p_ReportSPITObject) == false) {					
						return;
					}
				}
				
				ptrSPITObject = new SPITObject();
				boolNew = true;
			}
			else {
				if (m_SPITClientInterface != NULL) {
					//The user's app my veto the SPITType
					if (m_SPITClientInterface->checkSPITObjectVeto(p_ReportSPITObject) == false) {	
						//SpitObject exists and User APp denied the SPITObject -> remove the SPITObject without sending a message
						SPITObject::removeSPITObject(ptrSPITObject);
						delete ptrSPITObject;
						return;
					}
				}
				
			}
		}
		else {
			boolRemote = true;
			ptrSPITObject = SPITObject::getSPITObjectRemote(p_ReportSPITObject->getID());
			ptrSPITType = SPITType::getSPITTypeRemote(p_ReportSPITObject->getSPITTypeID());
			if (ptrSPITType == NULL) {
				//No associated SPITType found
				return;
			}
			if (ptrSPITObject == NULL) {
				ptrSPITObject = new SPITObject();
				boolNew = true;
			}
		}
		
		
		try {
			boolChanged = ptrSPITObject->setMessageOP(p_ReportSPITObject);
		}
		catch (...) {
			//setMessageOP error - the SPITObject can't be changed, it may be fixed and crated by this client
			sendSPITObjectInternal(ptrSPITObject);
			return;
		}
		
		if (boolRemote == false) {
			//set the virtual action assignment
			ptrSPITObject->setActionAssignedType(ptrSPITType->getActionAssigned());
			SPITObject::addSPITObject(ptrSPITObject);
		}
		else {
			SPITObject::addSPITObjectRemote(ptrSPITObject);
		}
		
		NotifyReportSPITObject* o_Runnable = new NotifyReportSPITObject(boolRemote, boolNew, m_SPITClientInterface, *ptrSPITType, *ptrSPITObject);
		SPIT_Handler::addRunnable(o_Runnable);
		
		if (ptrSPITObject->setConfirmed(true)  == true || boolChanged == true) {
			//Confirmation has changed
			sendSPITObjectInternal(ptrSPITObject);
		}
		
	}
}
void SPIT_Client::parseRemoveSPITObject(SPIT_Message::RemoveSPITObject* p_RemoveSPITObject) {
//printf("parse RemoveSPITObject %s\n", p_RemoveSPITObject->getID());
	if (p_RemoveSPITObject == NULL) return;
	bool boolRemote = false;
	SPITObject* ptrSPITObject;
	SPITType* ptrSPITType;	
	
	
	{
		unique_lock<recursive_mutex> o_LockSPITObject(*SPITObject::getMutex());
		unique_lock<recursive_mutex> o_LockSPITType(*SPITType::getMutex());

		ptrSPITObject = SPITObject::getSPITObjectRemote(p_RemoveSPITObject->getID());
		if (ptrSPITObject != NULL) {
			boolRemote = true;
			ptrSPITType = SPITType::getSPITTypeRemote(ptrSPITObject->getSPITTypeID());
			if (ptrSPITType != NULL) {
				NotifyRemoveSPITObject* o_Runnable = new NotifyRemoveSPITObject(boolRemote, m_SPITClientInterface, *ptrSPITType, *ptrSPITObject);
				SPIT_Handler::addRunnable(o_Runnable);
			}
			else {
				//No associated SPITType found
				//delete the SPITObject without a notification
			}
			SPITObject::removeSPITObjectRemote(ptrSPITObject);
			delete ptrSPITObject;
		}
		else {
			//call removeSPITObject, this will notify the server that the object is removed
			removeSPITObject(p_RemoveSPITObject->getID());
		}		
		

/*		ptrSPITObject = SPITObject::getSPITObjectRemote(p_RemoveSPITObject->getID());
		if (ptrSPITObject != NULL) {
			boolRemote = true;
			ptrSPITType = SPITType::getSPITTypeRemote(ptrSPITObject->getSPITTypeID());
		}
		else {
			ptrSPITObject = SPITObject::getSPITObject(p_RemoveSPITObject->getID());
			if (ptrSPITObject != NULL) {
				boolRemote = false;
				ptrSPITType = SPITType::getSPITType(ptrSPITObject->getSPITTypeID());
			}
			else {
				return;
			}
		}
		if (ptrSPITType != NULL) {
			NotifyRemoveSPITObject* o_Runnable = new NotifyRemoveSPITObject(boolRemote, m_SPITClientInterface, *ptrSPITType, *ptrSPITObject);
			SPIT_Handler::addRunnable(o_Runnable);
		}
		else {
			//No associated SPITType found
			//delete the SPITObject without a notification
		}
		
		if (boolRemote == false) {
			SPITObject::removeSPITObject(ptrSPITObject);
		}
		else {
			SPITObject::removeSPITObjectRemote(ptrSPITObject);
		}
		delete ptrSPITObject;
 */
	}	
}

int SPIT_Client::sendSPITObjectInternal(SPITObject* ptrSPITObject) {
	if (ptrSPITObject == NULL) return -1;
	SPIT_Message::ReportSPITObject* ptrSMReportSPITObject;
	ptrSMReportSPITObject = new SPIT_Message::ReportSPITObject();
	ptrSPITObject-> fillMessageOP(ptrSMReportSPITObject);	

	return sendSPITMessage(ptrSMReportSPITObject);
}
/**
 * When the client changes an existing SPITObject (normally the client should not do this), then he should send the changes to the SPITServer.<br>
 * An eventSPITObject occurs.<br>
 * The SPITServer may discard the changes and a second eventSPITObject occurs to correct the changes.<br>
 * @param p_SPITObject the pointer to the changed SPITObject.<br>
 * CAUTION: The SPITObject must already exist. 
 * @return true = SPITObject is changed, false = not changed, maybe p_SPITObject is NULL or the SPITObject doesn't exists
 */
bool SPIT_Client::changeSPITObject(SPITObject* p_SPITObject) {
	if (p_SPITObject == NULL) return false;
	SPITObject* ptrSPITObject;
	SPITType* ptrSPITType;
	{
		unique_lock<recursive_mutex> o_LockObject(*SPITObject::getMutex());
		unique_lock<recursive_mutex> o_LockType(*SPITType::getMutex());
		
		ptrSPITObject = SPITObject::getSPITObject(p_SPITObject->getID());
		if (ptrSPITObject == NULL) return false;
		ptrSPITType = SPITType::getSPITType(p_SPITObject->getID());
		if(ptrSPITObject == NULL) return false;
		
		ptrSPITObject->setActionAssigned(p_SPITObject->getActionAssigned());;
		ptrSPITObject->setDelay(p_SPITObject->getDelay());
		ptrSPITObject->setDuration(p_SPITObject->getDelay());
		ptrSPITObject->setFixed(p_SPITObject->getFixed());
		ptrSPITObject->setFrameFadeInLength(p_SPITObject->getFrameFadeInLength());
		ptrSPITObject->setFrameFadeOutLength(p_SPITObject->getFrameFadeOutLength());
		ptrSPITObject->setFrameLength(p_SPITObject->getFrameLength());
		ptrSPITObject->setFrameStart(p_SPITObject->getFrameStart());
		ptrSPITObject->setName(p_SPITObject->getName());
		ptrSPITObject->setRemark(p_SPITObject->getRemark());
		ptrSPITObject->setSPITTypeID(p_SPITObject->getSPITTypeID());
		ptrSPITObject->setServerParams(p_SPITObject->getServerParams());
		ptrSPITObject->setTrack(p_SPITObject->getTrack());
		ptrSPITObject->setValueFactor(p_SPITObject->getValueFactor());
		ptrSPITObject->setValueUse(p_SPITObject->getValueUse());

		NotifyReportSPITObject* o_Runnable = new NotifyReportSPITObject(false, false, m_SPITClientInterface, *ptrSPITType, *ptrSPITObject);
		SPIT_Handler::addRunnable(o_Runnable);

		sendSPITObjectInternal(ptrSPITObject);
	}
	return true;
}

/**
 * Is called when the SPITServer removes a SPITObject (parseRemoveSPITObject).<br>
 * When the client wants to remove a SPITObject with id == p_SPITObjectID (normally the client should not do this), then he should send it to the SPITServer.<br>
 * The corresponding SPITObject if listet in SPITObject will be deleted.<br>
 * An eventRemoveSPITType will occur.
 * If an connection to a SPITServer is established, the SPITServer may discard the deletion and an eventSPITObject occurs<br>
 *  -> the SPITObject is listet again.<br>
 * @param p_SPITObjectID the id of the SPITObject to remove.
 * @return 0 = sended to the SPITServer, -1 = not sended to the SPITServer or nos SPITObject with id == p_SPITObjectID exists.
  */
int SPIT_Client::removeSPITObject(string p_SPITObjectID) {
	if (p_SPITObjectID.length() <= 0) return -1;
	SPITType* ptrSPITType;
	SPITObject* ptrSPITObject;
	SPIT_Message::RemoveSPITObject* ptrSMRemoveSPITObject;
	{
		unique_lock<recursive_mutex> o_LockObject(*SPITObject::getMutex());
		unique_lock<recursive_mutex> o_LockType(*SPITType::getMutex());
		
		
		ptrSPITObject = SPITObject::getSPITObject(p_SPITObjectID);
		if (ptrSPITObject == NULL) return -1;
		ptrSPITType = SPITType::getSPITType(ptrSPITObject->getSPITTypeID());
		if (ptrSPITType == NULL) return -1;
		ptrSMRemoveSPITObject = new SPIT_Message::RemoveSPITObject();
		ptrSPITObject->fillMessageOP(ptrSMRemoveSPITObject);
		NotifyRemoveSPITObject* o_Runnable = new NotifyRemoveSPITObject(false, m_SPITClientInterface, *ptrSPITType, *ptrSPITObject);
		SPIT_Handler::addRunnable(o_Runnable);
		SPITObject::removeSPITObject(ptrSPITObject);
		delete ptrSPITObject;
	}
	return sendSPITMessage(ptrSMRemoveSPITObject);
}


//----- Play SPIT Objects ------

/*
void SPIT_Client::parsePlaySPITStart(SPIT_Message::PlaySPITStart* p_PlaySPITStart) {
//	printf("parse PlaySPITStart ID: %s Name: %s Running: %d\n", p_PlaySPITStart->getPlayerID(), p_PlaySPITStart->getPlayerName(), p_PlaySPITStart->getPlayerIsRunning());
	SPITPlayer* ptrSPITPlayer;
	{
		unique_lock<recursive_mutex> o_LockSPITPlayer(*SPITPlayer::getMutex());
		ptrSPITPlayer = SPITPlayer::getSPITPlayer(p_PlaySPITStart->getPlayerID());
		if (ptrSPITPlayer == NULL) return;
		ptrSPITPlayer->setIsRunning(p_PlaySPITStart->getPlayerIsRunning());
		ptrSPITPlayer->setName(p_PlaySPITStart->getPlayerName());
		ptrSPITPlayer->setType(p_PlaySPITStart->getPlayerType());
		ptrSPITPlayer->setTime(p_PlaySPITStart->getTime());
		ptrSPITPlayer->startPlaySequence();

		NotifyPlaySPITStart* o_Runnable = new NotifyPlaySPITStart(m_SPITClientInterface, ptrSPITPlayer->getID(), ptrSPITPlayer->getName(), ptrSPITPlayer->getType(), ptrSPITPlayer->getIsRunning(), ptrSPITPlayer->getTime());
		SPIT_Handler::addRunnable(o_Runnable);
	}
}
void SPIT_Client::parsePlaySPITObject(SPIT_Message::PlaySPITObject* p_PlaySPITObject) {
//	printf("parse RPlaySPITObject Object:%s\n", p_PlaySPITObject->getSPITObjectID());
	SPITPlayer* ptrSPITPlayer;
	SPITPlayer::PlayObject* ptrPlayObject;
	SPITObject* ptrSPITObject;
	
	{
		unique_lock<recursive_mutex> o_LockSPITPlayer(*SPITPlayer::getMutex());
		
		ptrSPITPlayer = SPITPlayer::getSPITPlayer(p_PlaySPITObject->getPlayerID());
		if (ptrSPITPlayer == NULL) return;
		ptrPlayObject = ptrSPITPlayer->addSPITObject_Playing(p_PlaySPITObject->getSPITObjectID());

		//Fetch infoemations from SPITObject
		{
			unique_lock<recursive_mutex> o_LockSPITObject(*SPITObject::getMutex());

			ptrSPITObject = SPITObject::getSPITObject(p_PlaySPITObject->getSPITObjectID());
			if (ptrSPITObject == NULL) {
				return;
			}
			if (ptrPlayObject == NULL) {
				return;
			}
			ptrPlayObject->setPlayerIsRunning(ptrSPITPlayer->getIsRunning());
			ptrPlayObject->setFrameInside(p_PlaySPITObject->getFrameInsideObject(), p_PlaySPITObject->getPlayPrePosition());
			ptrPlayObject->setValue(p_PlaySPITObject->getValue(), p_PlaySPITObject->getPlayPrePosition());
			ptrPlayObject->setValueFactor(ptrSPITObject->getValueFactor());
			ptrPlayObject->setValueFlag(p_PlaySPITObject->getValueFlag());
		}
	}
	
}
void SPIT_Client::parsePlaySPITEnd(SPIT_Message::PlaySPITEnd* p_PlaySPITEnd) {
//	printf("parse PlaySPITEnd \n");
	SPITPlayer* ptrSPITPlayer;
	SPITType* ptrSPITType = NULL;
	SPITObject* ptrSPITObject = NULL;
	{
		unique_lock<recursive_mutex> o_LockSPITPlayer(*SPITPlayer::getMutex());
	
		ptrSPITPlayer = SPITPlayer::getSPITPlayer(p_PlaySPITEnd->getPlayerID());
		if (ptrSPITPlayer == NULL) return;

		unordered_map<string, SPITPlayer::PlayObject*>* ptrPlayObjects;
		SPITPlayer::PlayObject* ptrPlayObject;
		bool boolStarted;
		bool boolStopped;

		ptrPlayObjects = ptrSPITPlayer->getPlayObjects();
		for (auto i = ptrPlayObjects->begin(); i != ptrPlayObjects->end(); i++) {
			ptrPlayObject = i->second;
			boolStarted = false;
			boolStopped = false;
			if (ptrPlayObject->getIsNew()) {
				boolStarted = true;
			}
			if (ptrPlayObject->getIsPlayed() == false) {
				boolStopped = true;
			}
			{
				unique_lock<recursive_mutex> o_LockSPITObject(*SPITObject::getMutex());
				unique_lock<recursive_mutex> o_LockSPITType(*SPITType::getMutex());
				
				ptrSPITObject = SPITObject::getSPITObject(ptrPlayObject->getSPITObjectID());
				if (ptrSPITObject == NULL) {
					//set isplayed to false, the ptrPlayObject will be removed by ptrSPITPlayer->endPlaySequence()
					ptrPlayObject->setIsPlayed(false);
					continue;
				}
				ptrSPITType = SPITType::getSPITType(ptrSPITObject->getSPITTypeID());
				if (ptrSPITType == NULL) {
					//set isplayed to false, the ptrPlayObject will be removed by ptrSPITPlayer->endPlaySequence()
					ptrPlayObject->setIsPlayed(false);
					continue;
				}
				NotifyPlaySPITObject* o_Runnable = new NotifyPlaySPITObject(boolStarted, boolStopped, m_SPITClientInterface, 
						*ptrSPITType, *ptrSPITObject, *ptrPlayObject);
				SPIT_Handler::addRunnable(o_Runnable);
			}
		}
		ptrSPITPlayer->endPlaySequence();
	}
}
*/


void SPIT_Client::parsePlaySPITStart(SPIT_Message::PlaySPITStart* p_PlaySPITStart) {
//	printf("parse PlaySPITStart ID: %s Name: %s Running: %d\n", p_PlaySPITStart->getPlayerID(), p_PlaySPITStart->getPlayerName(), p_PlaySPITStart->getPlayerIsRunning());
	SPITPlayer* ptrSPITPlayer;
	{
		unique_lock<recursive_mutex> o_LockSPITPlayer(*SPITPlayer::getMutex());
		ptrSPITPlayer = SPITPlayer::getSPITPlayer(p_PlaySPITStart->getPlayerID());
		if (ptrSPITPlayer == NULL) return;
		ptrSPITPlayer->setIsRunning(p_PlaySPITStart->getPlayerIsRunning());
		ptrSPITPlayer->setName(p_PlaySPITStart->getPlayerName());
		ptrSPITPlayer->setType(p_PlaySPITStart->getPlayerType());
		ptrSPITPlayer->setTime(p_PlaySPITStart->getTime());
		ptrSPITPlayer->startPlaySequence();

		NotifyPlaySPITStart* o_Runnable = new NotifyPlaySPITStart(m_SPITClientInterface, ptrSPITPlayer->getID(), ptrSPITPlayer->getName(), ptrSPITPlayer->getType(), ptrSPITPlayer->getIsRunning(), ptrSPITPlayer->getTime());
		SPIT_Handler::addRunnable(o_Runnable);
	}
}
void SPIT_Client::parsePlaySPITObject(SPIT_Message::PlaySPITObject* p_PlaySPITObject) {
//	printf("parse RPlaySPITObject Object:%s\n", p_PlaySPITObject->getSPITObjectID());
	SPITPlayer* ptrSPITPlayer;
	SPITPlayer::PlaySPITType* ptrPlaySPITType;
	SPITObject* ptrSPITObject;
	
	{
		unique_lock<recursive_mutex> o_LockSPITPlayer(*SPITPlayer::getMutex());
		
		ptrSPITPlayer = SPITPlayer::getSPITPlayer(p_PlaySPITObject->getPlayerID());
		if (ptrSPITPlayer == NULL) return;

		//Fetch infoemations from SPITObject
		{
			unique_lock<recursive_mutex> o_LockSPITObject(*SPITObject::getMutex());

			ptrSPITObject = SPITObject::getSPITObject(p_PlaySPITObject->getSPITObjectID());
			if (ptrSPITObject == NULL) {
				return;
			}
			ptrPlaySPITType = ptrSPITPlayer->addSPITType_Playing(ptrSPITObject->getSPITTypeID());
			if (ptrPlaySPITType == NULL) {

				return;
			}
			ptrPlaySPITType->setPlayerIsRunning(ptrSPITPlayer->getIsRunning());
			ptrPlaySPITType->addValue(((double)p_PlaySPITObject->getValue()/1000000), ((double)ptrSPITObject->getValueFactor()/1000000));
		}
	}
	
}
void SPIT_Client::parsePlaySPITEnd(SPIT_Message::PlaySPITEnd* p_PlaySPITEnd) {
//	printf("parse PlaySPITEnd \n");
	SPITPlayer* ptrSPITPlayer;
	SPITType* ptrSPITType = NULL;
	{
		unique_lock<recursive_mutex> o_LockSPITPlayer(*SPITPlayer::getMutex());
	
		ptrSPITPlayer = SPITPlayer::getSPITPlayer(p_PlaySPITEnd->getPlayerID());
		if (ptrSPITPlayer == NULL) return;

		unordered_map<string, SPITPlayer::PlaySPITType*>* ptrPlaySPITTypes;
		SPITPlayer::PlaySPITType* ptrPlaySPITType;
		bool boolStarted;
		bool boolStopped;

		ptrPlaySPITTypes = ptrSPITPlayer->getPlaySPITTypes();
		for (auto i = ptrPlaySPITTypes->begin(); i != ptrPlaySPITTypes->end(); i++) {
			ptrPlaySPITType = i->second;
			boolStarted = false;
			boolStopped = false;
			if (ptrPlaySPITType->getIsNew()) {
				boolStarted = true;
			}
			if (ptrPlaySPITType->getIsPlayed() == false) {
				boolStopped = true;
			}
			{
				unique_lock<recursive_mutex> o_LockSPITObject(*SPITObject::getMutex());
				unique_lock<recursive_mutex> o_LockSPITType(*SPITType::getMutex());
				
				ptrSPITType = SPITType::getSPITType(ptrPlaySPITType->getSPITTypeID());
				if (ptrSPITType == NULL) {
					//set isplayed to false, the ptrPlayObject will be removed by ptrSPITPlayer->endPlaySequence()
					ptrPlaySPITType->setIsPlayed(false);
					continue;
				}
				NotifyPlaySPITType* o_Runnable = new NotifyPlaySPITType(ptrPlaySPITType->getPlayerID(), 
						boolStarted, boolStopped, ptrPlaySPITType->getValue(), ptrPlaySPITType->getPlayerIsRunning(), 
						m_SPITClientInterface, ptrPlaySPITType->getSPITTypeID());
				SPIT_Handler::addRunnable(o_Runnable);
			}
		}
		ptrSPITPlayer->endPlaySequence();
	}
}




void SPIT_Client::parseReportPlayStatus(SPIT_Message::ReportPlayStatus* p_ReportPlayStatus) {
	//only REMOTE server -> client
	//printf(" parseReportPlayStatus ");
	SPITType* ptrSPITType = NULL;
	SPITObject* ptrSPITObject = NULL;
	{
		unique_lock<recursive_mutex> o_LockSPITObject(*SPITObject::getMutex());
		unique_lock<recursive_mutex> o_LockSPITType(*SPITType::getMutex());

		ptrSPITObject = SPITObject::getSPITObjectRemote(p_ReportPlayStatus->getID());
		if (ptrSPITObject == NULL) {
			ptrSPITObject = SPITObject::getSPITObject(p_ReportPlayStatus->getID());
			if (ptrSPITObject == NULL) {
				return;
			}
		}
		if (ptrSPITObject->getServerRemote() == false) {
			ptrSPITType = SPITType::getSPITType(ptrSPITObject->getSPITTypeID());
		}
		else {
			ptrSPITType = SPITType::getSPITTypeRemote(ptrSPITObject->getSPITTypeID());
		}
		if (ptrSPITType == NULL) {
			return;
		}
		NotifyReportPlayStatus* o_Runnable = new NotifyReportPlayStatus(m_SPITClientInterface, *ptrSPITType, *ptrSPITObject, p_ReportPlayStatus->getFrameInsideObject(), p_ReportPlayStatus->getValue());
		SPIT_Handler::addRunnable(o_Runnable);
	}
}

	//--------- Remote ----------

/**
 * Send a remote control command to the SPITServer.<br>
 * There may be different command types:
 * a) Some don't use value e.g. go to next scene, olay a jingle
 * b) Some use values e.g. the main sound fader
 * @param p_SPITObjectID the id of the remote SPITObject (this represents the remote control command)
 * @param p_Valueppm the value in ppm <br>
 * in case a) the value is ignored<br>
 * in case b) the value is used
 * @param p_ValueFlag indicates how the server should use p_Valueppm:<br>
 * VALUEFLAG_DIRECT (0x00) The server overwrites its existing value with p_Value<br>
 * VALUEFLAG_INCREASE (0x01) The server increases his existing value by p_Valueppm
 * VALUEFLAG_DECREASE (0x01) The server decreases his existing value by p_Valueppm
 */
void SPIT_Client::sendRemoteObject(string p_SPITObjectID, int p_Valueppm, char p_ValueFlag) {
	SPITObject* ptrRemoteSPITObject;
	SPIT_Message::PlaySPITStart* ptrPlaySPITStart;
	SPIT_Message::PlaySPITObject* ptrPlaySPITObject;
	SPIT_Message::PlaySPITEnd* ptrPlaySPITEnd;
	
	{
		unique_lock<recursive_mutex> o_LockRemote(*SPITObject::getMutex());
		
		ptrRemoteSPITObject = SPITObject::getSPITObjectRemote(p_SPITObjectID);
		if (ptrRemoteSPITObject == NULL) return;
		
		ptrPlaySPITStart = new SPIT_Message::PlaySPITStart();
		ptrPlaySPITStart->setPlayerID(m_ID);
		ptrPlaySPITStart->setPlayerName(m_SPITClientInterface->getClientProjectName());
		ptrPlaySPITStart->setPlayerIsRunning(false);
		ptrPlaySPITStart->setPlayerType(PLAYERTYPE_DISCRETE);
		ptrPlaySPITStart->setTime(0);
		sendSPITMessage(ptrPlaySPITStart);
		
		ptrPlaySPITObject = new SPIT_Message::PlaySPITObject();
		ptrPlaySPITObject->setID(ptrRemoteSPITObject->getID());
		ptrPlaySPITObject->setPlayPrePosition(false);
		ptrPlaySPITObject->setPlayerID(m_ID);
		ptrPlaySPITObject->setValue(p_Valueppm);
		ptrPlaySPITObject->setValueFlag(p_ValueFlag);
		sendSPITMessage(ptrPlaySPITObject);
		
		ptrPlaySPITEnd = new SPIT_Message::PlaySPITEnd();
		ptrPlaySPITEnd->setPlayerID(m_ID);
		sendSPITMessage(ptrPlaySPITEnd);
		
	}
}

