/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

/* 
 * Unix implementation of a SPITSocket.
 * 
 * File:   UnixSocket.cpp
 * Author: Hagi
 * 
 * Created on 12. April 2019, 15:58
 */



#include "_SPITMACROS.h"
#if defined(OS_MAC) || defined(OS_LINUX)


#include "UnixSocket.h"

UnixSocket::UnixSocket() : SPITSocket(){
	SPITSocket::FAMILY_IP4 = AF_INET;
	SPITSocket::FAMILY_IP6 = AF_INET6;
	m_UDPSocketIP4 = INVALID_SOCKET;
	m_UDPSocketIP6 = INVALID_SOCKET;
	createSocket_UDP();
}


UnixSocket::~UnixSocket() {
	
	int intResultIP4;
	int intResultIP6;
	if (m_UDPSocketIP4 != INVALID_SOCKET) {
//		intResultIP4 = shutdown(m_UDPSocketIP4, 2);
		intResultIP4 = close(m_UDPSocketIP4);
		m_UDPSocketIP4 = INVALID_SOCKET;
	}
	if (m_UDPSocketIP6 != INVALID_SOCKET) {
//		intResultIP6 = shutdown(m_UDPSocketIP6, 2);
		intResultIP6 = close(m_UDPSocketIP6);
		m_UDPSocketIP6 = INVALID_SOCKET;
	}
	this_thread::sleep_for(chrono::milliseconds(100));
}

int UnixSocket::init() {
    return 0;
}

int UnixSocket::cleanUp() {
    return 0;
}

/**
 * Creates two new sockets.<br>
 * One socket for IP4 and one for IP6.<br>
 * If there exist some, they will be closed.<br>
 * @return 
 */
bool UnixSocket::createSocket_UDP() {
	bool boolSuccess = true;
	int intResult;
	int intOptionValue4 = 1;
	int intOptionValue6 = 1;
	u_char charOptionValue = 0x01;
	
	//ip4
	if (m_UDPSocketIP4 != INVALID_SOCKET) {
		close(m_UDPSocketIP4);
		m_UDPSocketIP4 = INVALID_SOCKET;
	}
	if (m_UDPSocketIP6 != INVALID_SOCKET) {
		close(m_UDPSocketIP6);
		m_UDPSocketIP6 = INVALID_SOCKET;
	}

	
	m_UDPSocketIP4 = socket(AF_INET,SOCK_DGRAM, 0);
	if (m_UDPSocketIP4 == INVALID_SOCKET) {
		boolSuccess = false;
	}
	else {
		intResult = setsockopt(m_UDPSocketIP4, SOL_SOCKET, SO_REUSEADDR, (void*)&intOptionValue4, sizeof(intOptionValue4));
	}
	if (intResult < 0) {
		boolSuccess = false;
	}
	
	m_UDPSocketIP6 = socket(AF_INET6,SOCK_DGRAM, 0);
	if (m_UDPSocketIP6 == INVALID_SOCKET) {
		boolSuccess = false;
		close(m_UDPSocketIP4);
		m_UDPSocketIP4 = INVALID_SOCKET;
	}
	else {
		intResult = setsockopt(m_UDPSocketIP6, SOL_SOCKET, SO_REUSEADDR, (void*)&intOptionValue6, sizeof(intOptionValue6));
	}
	if (intResult < 0) {
		boolSuccess = false;
	}
	//set the m_UDPSocketIP4 and m_UDPSocketIP6 in NON BLOCKING mode because
	//if the UDPSockets are in blocking mode the receiveMessage does not return in every constellation  
	if (boolSuccess == true) {
		//increase the buffer size of the socket
		int intBufferSize = 2046 * 512;
		intResult = setsockopt(m_UDPSocketIP4, SOL_SOCKET, SO_RCVBUF, (void*)&intBufferSize, sizeof(intBufferSize));
		intResult = setsockopt(m_UDPSocketIP4, SOL_SOCKET, SO_SNDBUF, (void*)&intBufferSize, sizeof(intBufferSize));
		intResult = setsockopt(m_UDPSocketIP6, SOL_SOCKET, SO_RCVBUF, (void*)&intBufferSize, sizeof(intBufferSize));
		intResult = setsockopt(m_UDPSocketIP6, SOL_SOCKET, SO_SNDBUF, (void*)&intBufferSize, sizeof(intBufferSize));
	}
	if (boolSuccess == true) {
		int intFlags;
		intFlags = fcntl(m_UDPSocketIP4, F_GETFL);
		fcntl(m_UDPSocketIP4, F_SETFL, intFlags | O_NONBLOCK);
		intFlags = fcntl(m_UDPSocketIP6, F_GETFL);
		fcntl(m_UDPSocketIP6, F_SETFL, intFlags | O_NONBLOCK);
	}
	return boolSuccess;
}

/**
 * Binds the socket to the client's ip address and port.<br>
 * The socket option SO_REUSEADDR is set to true.<br>
 * @param p_ClientIP ip address of the client
 * @param p_ClientPort port of the client.<br>
 * if p_Port == "0" any free port will be used<br>
 * @return -1 id bind fails, o if bind is ok
 * 
 */
int UnixSocket::bindSocket(string p_ServerIP, string p_ClientIP, string p_ClientPort) {
    addrinfo *clientAI;
	addrinfo *serverAI;
	struct sockaddr_in* ptrIP4SocketAddress;
	struct sockaddr_in o_IP4SocketAddress;
	struct sockaddr_in6* ptrIP6SocketAddress;
	struct sockaddr_in6* ptrIP6SocketAddress_Server;
	struct sockaddr_in6 o_IP6SocketAddress;
	int intSocklen;
	string stringClientPort;
	if (p_ClientPort.length() <= 0) stringClientPort = "0";
	else stringClientPort = p_ClientPort;
	//int intResult = SOCKET_ERROR;
	int intResult = 0;
	if (getaddrinfo(p_ClientIP.data(), p_ClientPort.data(), NULL, &clientAI) != 0) {
		return -1;
	}
	switch(clientAI->ai_family) {
		case AF_INET:
			if (m_UDPSocketIP4 == INVALID_SOCKET) {
				break;
			}
			ptrIP4SocketAddress = reinterpret_cast<sockaddr_in*>(clientAI->ai_addr);
			intSocklen = sizeof (*ptrIP4SocketAddress);	
			//copy the ptrIP4SocketAddress, may be not filled correct by getaddrinfo
			memset(&o_IP4SocketAddress, 0, intSocklen);
			o_IP4SocketAddress.sin_addr = ptrIP4SocketAddress->sin_addr;
			o_IP4SocketAddress.sin_family = ptrIP4SocketAddress->sin_family;
                        #if defined(OS_MAC)  
                	o_IP4SocketAddress.sin_len = ptrIP4SocketAddress->sin_len;
                        #endif
                        #if defined(OS_LINUX)
                	//o_IP4SocketAddress.sin_zero = ptrIP4SocketAddress->sin_zero;
                        #endif
			o_IP4SocketAddress.sin_port = ptrIP4SocketAddress->sin_port;
			intResult = bind(m_UDPSocketIP4, (const sockaddr*)(ptrIP4SocketAddress),(socklen_t)intSocklen);
			if (intResult < 0) {
				break; //fe80:0:0:0:28e:d6ef:8971:7871%9
			}
			setBoundClientFamily(AF_INET);
			if (intResult == 0) {
				if (getsockname(m_UDPSocketIP4, (sockaddr*)ptrIP4SocketAddress, (socklen_t*)&intSocklen) != -1) {
					setBoundClientPort(to_string((int)ptrIP4SocketAddress->sin_port));
				}
			}
			break;
		case AF_INET6:
			if (m_UDPSocketIP6 == INVALID_SOCKET) {
				break;
			}
			ptrIP6SocketAddress = reinterpret_cast<sockaddr_in6*>(clientAI->ai_addr);
			//set the scope_id of the client to the scope_id of the server
			//the scope id is the index of the network interface, which has received the udp-datagramm from the server and to whisch the socket will be bound
			//the p_ClientIP contains the IP6-address as the server has received from the client (scope id is actual the network interface index of the server's environment)
/*			if (getaddrinfo(p_ServerIP.data(), "0", NULL, &serverAI) != 0) {
				return -1;
			}			
			ptrIP6SocketAddress_Server = reinterpret_cast<sockaddr_in6*>(serverAI->ai_addr);
			
			ptrIP6SocketAddress->sin6_scope_id = ptrIP6SocketAddress_Server->sin6_scope_id;
			freeaddrinfo(serverAI);
*/			
			intSocklen = sizeof (*ptrIP6SocketAddress);
			
			//copy the ptrIP6SocketAddress, may be not filled correct by getaddrinfo
			o_IP6SocketAddress.sin6_addr = ptrIP6SocketAddress->sin6_addr;
			o_IP6SocketAddress.sin6_family = ptrIP6SocketAddress->sin6_family;
			o_IP6SocketAddress.sin6_flowinfo = ptrIP6SocketAddress->sin6_flowinfo;
			o_IP6SocketAddress.sin6_port = ptrIP6SocketAddress->sin6_port;
			o_IP6SocketAddress.sin6_scope_id = ptrIP6SocketAddress->sin6_scope_id;
			ptrIP6SocketAddress->sin6_scope_id = 2; //Scope link local = does not leave the subnet
			intResult = bind(m_UDPSocketIP6, (const sockaddr*)(ptrIP6SocketAddress), (socklen_t)intSocklen);
			if (intResult < 0) {
				int intError = errno;
				string stringError = strerror(intError);
				break;
			}
			setBoundClientFamily(AF_INET6);
			
			if (intResult == 0) {
				if (getsockname(m_UDPSocketIP6, (sockaddr*)ptrIP6SocketAddress, (socklen_t*)&intSocklen) != -1) {
					setBoundClientPort(to_string((int)ptrIP6SocketAddress->sin6_port));
				}				
			}
			break;
		default:
			setBoundClientFamily(0);
			intResult = -1;
			break;
	}
	freeaddrinfo(clientAI);
	
	if (intResult == -1) {
		setIsBound(false);
	}
	else {
		setIsBound(true);
	}
	return intResult;
}

/**
 * Set the sockopt SO_BROADCAST to 0 or 1
 * @param p_Broadcast true SO_BROADCAST is set<br>
 * false = SO_BROADCAST is not set
 * @return 0 = ok, -1 = error
 */
int UnixSocket::setBroadcast(bool p_Broadcast) {
	int intResult;
	int intOptionValue = 1;
	u_char charOptionValue = 0x01;
	//IP4
	if (p_Broadcast) {
		intOptionValue = 1;
	}
	else {
		intOptionValue = 0;
	}
	intResult = setsockopt(m_UDPSocketIP4, SOL_SOCKET, SO_BROADCAST, &intOptionValue, sizeof(int));
//	intResult = setsockopt(m_UDPSocketIP4, IPPROTO_IP, IP_MULTICAST_LOOP, (void*)&charOptionValue, sizeof(charOptionValue));

	//IP6
//	intResult = setsockopt(m_UDPSocketIP6, IPPROTO_IPV6, IPV6_MULTICAST_LOOP, (void*)&charOptionValue, sizeof(charOptionValue));
	
	return intResult;
}
/**
 * 
 * @param p_InetFamily AF_INET or AF_INET6
 * @param p_ServerIP the ip address of the server
 * @param p_ServerPort the port of the server
 * @param p_MessageBytes buffer containing the message 
 * @param p_MessageLength the length of the message
 * @return -1 send fails<br>
 * 0 send is done
 */
int UnixSocket::sendMessageTo(int p_InetFamily, string p_ServerIP, string p_ServerPort, const char* p_MessageBytes, int p_MessageLength) {
	if (m_UDPSocketIP4 == INVALID_SOCKET && m_UDPSocketIP6 == INVALID_SOCKET) return -1;
	struct addrinfo* ptrAddressInfo;
	struct sockaddr_in* ptrIP4SocketAddress;
	struct sockaddr_in o_IP4SocketAddress;
	struct sockaddr_in6* ptrIP6SocketAddress;
	struct sockaddr_in6 o_IP6SocketAddress;
	int intSocklen;
	int intResult = -1;

	// prepare AddressInfo
	if (getaddrinfo(p_ServerIP.data(), p_ServerPort.data(), NULL, &ptrAddressInfo) != 0) {
	    return -1;
	}
	switch(ptrAddressInfo->ai_family) {
		case AF_INET:
			ptrIP4SocketAddress = reinterpret_cast<sockaddr_in*>(ptrAddressInfo->ai_addr);
			intSocklen = sizeof(*ptrIP4SocketAddress);
			if (m_UDPSocketIP4 != INVALID_SOCKET) {
				//copy the ptrIP4SocketAddress, may be not filled correct by getaddrinfo
				memset(&o_IP4SocketAddress, 0, intSocklen);
				o_IP4SocketAddress.sin_addr = ptrIP4SocketAddress->sin_addr;
				o_IP4SocketAddress.sin_family = ptrIP4SocketAddress->sin_family;
				o_IP4SocketAddress.sin_port = ptrIP4SocketAddress->sin_port;
				intResult = sendto(m_UDPSocketIP4, p_MessageBytes, p_MessageLength, 0, (const sockaddr*)&o_IP4SocketAddress, (socklen_t)intSocklen);
//if (intResult < 0) printf("UnixSocket::sendMessageTo IP4 sendto %s FAILED %s\n", p_ServerIP.data(), strerror(errno));
				freeaddrinfo(ptrAddressInfo);
			}
			break;
		case AF_INET6:
			ptrIP6SocketAddress = reinterpret_cast<sockaddr_in6*>(ptrAddressInfo->ai_addr);			
			intSocklen = sizeof(*ptrIP6SocketAddress);
			if (m_UDPSocketIP6 != INVALID_SOCKET) {
				//copy the ptrIP6SocketAddress, may be not filled correct by getaddrinfo
				o_IP6SocketAddress.sin6_addr = ptrIP6SocketAddress->sin6_addr;
				o_IP6SocketAddress.sin6_family = ptrIP6SocketAddress->sin6_family;
				o_IP6SocketAddress.sin6_flowinfo = ptrIP6SocketAddress->sin6_flowinfo;
				o_IP6SocketAddress.sin6_port = ptrIP6SocketAddress->sin6_port;
				o_IP6SocketAddress.sin6_scope_id = ptrIP6SocketAddress->sin6_scope_id;
				intResult = sendto(m_UDPSocketIP6, p_MessageBytes, p_MessageLength, 0, (const sockaddr*)&o_IP6SocketAddress, (socklen_t)intSocklen);
//if (intResult < 0) printf("UnixSocket::sendMessageTo IP6 sendto %s FAILED %s\n", p_ServerIP.data(), strerror(errno));
				freeaddrinfo(ptrAddressInfo);
			}
			break;
		default:
			intResult = -1;
	}
	
	if (intResult < 0) {
		return -1;
	}
    return 0;
}
/**
 * Starts receiving a message
 * @param p_InetFamily	AF_INET or AF_INET6
 * @param p_Buffer	a char* buffer large enough to hold the expected data 
 * @param p_BufferLength the length of the buffer
 * @param p_ServerIP = NULL, the ip address is not determined<br>
 * != NULL after receiving a message p_ServerID contains the ip address of the sender/server
 * @return 
 */
int UnixSocket::receiveMessage(int p_InetFamily, char* p_Buffer, int p_BufferLength, string* p_ServerIP) {
	int intResult = -1;
//	ADDRINFO *AI;
	struct sockaddr_in o_IP4SocketAddress;
	struct sockaddr_in6 o_IP6SocketAddress;
	int intSocklen;

	switch(p_InetFamily) {
		case AF_INET:
			if (m_UDPSocketIP4 == INVALID_SOCKET) {
				break;
			}
			intSocklen = sizeof(o_IP4SocketAddress);
			memset(&o_IP4SocketAddress, 0, intSocklen);
			intResult = recvfrom(m_UDPSocketIP4, (void*)p_Buffer, p_BufferLength, 0, (sockaddr*)&o_IP4SocketAddress, (socklen_t*)&intSocklen);
			if (intResult >= 0) {
				if (p_ServerIP != NULL) {
					char bufferIP4[INET_ADDRSTRLEN];
					inet_ntop(AF_INET, &(o_IP4SocketAddress.sin_addr), bufferIP4, INET_ADDRSTRLEN);
					*p_ServerIP = bufferIP4;
				}
			}
			break;
		case AF_INET6:
			if (m_UDPSocketIP6 == INVALID_SOCKET) {
				break;
			}
			intSocklen = sizeof(o_IP6SocketAddress);
			memset(&o_IP6SocketAddress, 0, intSocklen);
			intResult = recvfrom(m_UDPSocketIP6, (void*)p_Buffer, p_BufferLength, 0, (sockaddr*)&o_IP6SocketAddress, (socklen_t*)&intSocklen);
			if (intResult >= 0) {
				if (p_ServerIP != NULL) {
					char bufferIP6[INET6_ADDRSTRLEN];
					inet_ntop(AF_INET6, &(o_IP6SocketAddress.sin6_addr), bufferIP6, INET6_ADDRSTRLEN);
					//inet_ntop removes the scope_id (xxx%scope_id from the o_IP6SocketAddress.sin6_addr
					//--> add the scope_id to the p_ServerIP
					//the scope id is the index of the network-interface, which has received the message
					
					string stringServerIP;
					size_t sizeFound;
					stringServerIP = bufferIP6;
					sizeFound = stringServerIP.find("%");
					if (sizeFound != string::npos) {
						stringServerIP = stringServerIP.substr(0, sizeFound);
					}
					*p_ServerIP = stringServerIP + "%" + to_string(o_IP6SocketAddress.sin6_scope_id);
				}
			}
			break;
	}
	return intResult;
}

/**
 * Calculates the broadcast address from a ip4 address.<br>
 * @param p_IPAddress the ip4 address
 * @return returns the broadcast address
 */
string UnixSocket::getBroadcastIP4(SPITSocket::IPAddress* p_IPAddress) {
    struct in_addr broadcast;
	unsigned long ulIPAddress;
	unsigned long ulSubnetMask;
	
	ulIPAddress = inet_addr(p_IPAddress->getIPAddress().data());
	ulSubnetMask = inet_addr(p_IPAddress->getSubnetMask().data());
    broadcast.s_addr = ulIPAddress | ~ulSubnetMask;
	
	return inet_ntoa(broadcast);
}

void UnixSocket::getAdaptersInfo() {
	struct ifaddrs *ifaddr, *ifa;
        struct sockaddr_in *o_sockaddr_in;
//        struct sockaddr_in6 *o_sockaddr_in6;
        char buffer[64];
        string stringIP4;
        string stringSubnet;
        void *in_addr;
        void *in_subnet;
        
	resetFoundIPAddresses();
  
	if (getifaddrs(&ifaddr) == -1) {
		printf("UnixSocket::getAdaptersInfo getifaddrs < 0\n");
		return;
	}

    struct sockaddr* ptrSockAddr;
	for (ifa = ifaddr; ifa != NULL; ifa = ifa->ifa_next) {

            if (ifa->ifa_addr == NULL) break;

                switch (ifa->ifa_addr->sa_family) {
                    case AF_INET:
                        o_sockaddr_in = (struct sockaddr_in *)ifa->ifa_addr;
                        in_addr = &o_sockaddr_in->sin_addr;
                        o_sockaddr_in = (struct sockaddr_in *)ifa->ifa_netmask;
                        in_subnet = &o_sockaddr_in->sin_addr;
                        break;
                    case AF_INET6:
                        continue;
                        /*
                        o_sockaddr_in6 = (struct sockaddr_in6 *)ifa->ifa_addr;
                        in_addr = &o_sockaddr_in6->sin6_addr;
                        o_sockaddr_in6 = (struct sockaddr_in6 *)ifa->ifa_netmask;
                        in_subnet = &o_sockaddr_in6->sin6_addr;
                        break;
                        */
                    default:
                            continue;
                }
                if (inet_ntop(ifa->ifa_addr->sa_family, in_addr, buffer, sizeof(buffer))) {
                    stringIP4 = buffer;
                }
                else {
                    continue;
                }      
                if (inet_ntop(ifa->ifa_addr->sa_family, in_subnet, buffer, sizeof(buffer))) {
                    stringSubnet = buffer;
                }
                else {
                    continue;
                }
#ifdef DEBUGOUTPUT
printf("Networkadapter %s: %s  Subnet %s\n", ifa->ifa_name, stringIP4.data(), stringSubnet.data());
#endif'
       		addIPAddress(stringIP4, stringSubnet);
	}
	//IP4 loopback maybe not ok
#ifdef DEBUGOUTPUT
	printf("Networkinterfaces add/change loopback 127.0.0.1 subnet 255.255.255.0\n");
	printf("\n");
#endif
	
	addIPAddress("127.0.0.1", "255.255.255.0");

	freeifaddrs(ifaddr);
	
	removeNotFoundIPAddresses();	
}
#endif