/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

/* 
 * File:   MySPITApp.cpp
 * Author: Hagi
 * 
 * Created on 17. April 2019, 06:24 pm
 * Changed on 13. December 2022
 * 
 * Description:
 * A SPITClient defines actions (functions) and connects to a SPITServer.
 * The actions are assigned to SPITTypes.
 * The client may create its own SPITTypes, which are sended to the server or receives SPITTypes from the server.
 * Each SPIType can trigger exactly one action.
 * The server may use this SPITTypes to create and play SPITObjects, he can create several SPITObjects from one SPITType.
 * Each SPITObject is assigned exactly to one SPITType.
 * When the server plays a SPITObject, an event function (eventPlayObject) is called in the client.
 * The client can then trigger the action for the assigned SPITType. 
 * --------------------------------------------------------------------------------------------------
 * all event.. functions are executed by a single SPIT_Handler thread.
 * -> The event functions are called singlethreaded one after the other.
 * --------------------------------------------------------------------------------------------------
 */

#include "MyApp.h"
#include "main.h"



string	MyApp::s_ClientName = "MySpit";			//Name of this Client
string	MyApp::s_ClientProjectName = "MyProject";	//Name of the client'loaded project
string	MyApp::s_ClientProjectID = s_ClientProjectName;		//ID of the client's project - maybe the same as the name or a specific id





MyApp::MyApp() {
	//create a new SPIT_Client instance
	//the SPIT_Client will do the main work for you (receiving/sending SPIT_Messages ....)

	
	m_SPIT_Client = new SPIT_Client(this);
	
	//You need to give your application a descriptive name
	m_SPIT_Client->setName(s_ClientName);
	
	
	//The following members are set in the eventConnectionStatus
	m_ServerID = "";
	m_ServerIP = "";
	m_ServerComputerName = "";
	m_ServerName = "";
	m_ServerStepTime = -1;

	//The follwing members are set in eventProjectReport
	m_ServerProjectID = ""; 
	m_ServerProjectName = ""; 
	m_ServerFramesPerSecond = 44100;	//is reset in eventProjectReport.
										//perhaps you may save the value from the last connection and set it here
										//liveSHOWsoftware will use 44100 frames per second as time resolution
										//All times are given in frames, 
										//with this parameter you can convert the times into milliseconds/seconds ...
									 
	
	

	//Set the client project ID and name
	m_SPIT_Client->setClientProjectName(m_ClientProjectName);
	//Used to decide if a reconnection is tried after the connection is lost
	//see eventServerDetected and eventConnectionStatus
	
	//you my load a client project
//	loadClientProject();
	
	//start the ato detection of servers
	//erverAutoDetect maybe stopped in eventConnectionStatus
	startServerAutoDetect();

	
	
		
}


MyApp::~MyApp() {
	//To close the connection and all other stuff delete the SPIT_Client instance
	if (m_SPIT_Client != NULL) {
		m_SPIT_Client->disconnect();
		delete m_SPIT_Client;
	}
}

//Following functions fullfill the SPIT_ClientInterface
//They are needed when the server sends his project, see eventProjectReport
//and they should be set in loadClientProject
string MyApp::getClientProjectID() {
	return m_ClientProjectID;
}
string MyApp::getClientProjectName() {
	return m_ClientProjectName;
}

void MyApp::loadClientProject() {
	//----- close the actual client project -----
	//inform the SPIT_Client that the actual project will close
	//If there is actual no connection to the server nothing happens
	SPIT_Client* ptrSPIT_Client;
	
	ptrSPIT_Client = m_SPIT_Client;
	
	if (ptrSPIT_Client != NULL) {
		ptrSPIT_Client->clientProjectClose();
	}
	//close the project
	m_ClientProjectName = "";
	m_ClientProjectID = "";
	
	//----- load new client project -----
	//	    Add your loading code here
	//If your application is a small one using only a few fixed actions
	//you may keep the things simple and code all loading here

	m_ClientProjectName = s_ClientProjectName; //maybe change here the project name after loading
	m_ClientProjectID = s_ClientProjectID; //maybe the same as s_ClientProjectName

	if (ptrSPIT_Client != NULL) {
		ptrSPIT_Client->setClientProjectName(m_ClientProjectName);
	}

	
	//----- inform the SPIT_Client that a new project is loaded -----
	if (ptrSPIT_Client != NULL) {
		ptrSPIT_Client->clientProjectLoaded();
	}
	
}

SPIT_Client* MyApp::getSPIT_Client() {
	return m_SPIT_Client;
}
//-----------------------------------------
//---------- ServerAutoDetection ----------

void MyApp::startServerAutoDetect() {
	if (m_SPIT_Client != NULL) {
		m_SPIT_Client->startServerAutoDetect();
	}
}
void MyApp::stopServerAutoDetect() {
	if (m_SPIT_Client != NULL) {
		m_SPIT_Client->stopServerAutoDetect();
	}
}

//------------------------------------------------------
//----- All events are delegated to the SPIT_Handler.
//----- The follwoing event functions are called
//----- by the one thread s_HandlerThread of the SPIT_Handler
//------------------------------------------------------

/**
 * Is always called when a new SPIT server is found.<br>
 * For this you have to call SPIT_Client::startServerAutoDetect.<br>
 * @param p_DetectedServer a reference to a copy of a SAD_DetectedServer instance.<br>
 * After this function call the copy of the SAD_DetectedServer will be deleted!!<br>
 */
void MyApp::eventServerDetected(const SAD_DetectedServer& p_DetectedServer) {
	//fetch the infoemations from the p_DetectedServer
#if defined(DEBUGOUTPUT)
	printf("mySPITApp\n");
	printf("-----> SERVER DETECTED\n");
	printf(" Name: %s\n", p_DetectedServer.getName().data());
	printf(" ID: %s\n", p_DetectedServer.getID().data());
	printf(" Computer: %s\n", p_DetectedServer.getComputerName().data());
	printf(" IP: %s\n", p_DetectedServer.getIP().data());
	printf(" Port: %s\n", p_DetectedServer.getPort().data());
	printf(" Addition: %s\n", p_DetectedServer.getAdditions().data());
	printf(" ClientIP: %s\n", p_DetectedServer.getClientIP().data());
	printf("\n");
#endif
	tryConnection(p_DetectedServer);
}

/**
 * The project name of the detected server is used to decide if a connection is possible.<br>
 * @param p_DetectedServer 
 * @return 
 */
bool MyApp::checkPossibleConnection(const SAD_DetectedServer& p_DetectedServer) {
	string stringServerProjectName;
	
	stringServerProjectName = p_DetectedServer.getProjectName();
	
	
	if (stringServerProjectName.length() <= 0) {
		//if the server has no project name sended -> try connection
		//the login will 
		return false;
	}
	else {
		//Maybe check the servers project name to decide if the connection is accepted or not
		//return false
	}
	return true;
}

void MyApp::tryConnection(const SAD_DetectedServer& p_DetectedServer) {
	//fetch the infoemations from the p_DetectedServer
	bool boolPossibleConnect;
	bool boolSuccess;
	
	boolPossibleConnect = checkPossibleConnection(p_DetectedServer);
	
	if (boolPossibleConnect == false) return;
	
	// --------------------------------------
	// You have three Options to use the detected server
	// OPTION 1 Use the first server found
	// OPTION 2 Use the first server found that fulfills specific conditions
	// OPTION 3 take a copy of the p_DetectedServer and do connection later
	//		SAD_DetectedServer o_DetectedServer = p_DetectedServer;	//this will call the copy constructor 
	// ---------------------------------------

	//Excample use the first detected LOOPBACK server, means the server runs on the same computer as this client
	//if (p_DetectedServer.getIpIsLoopback() == true) { //omit the if condition if you want to use the first server found 
	
		boolSuccess = false;
		//Stop auto detection because a suitable server was found
		stopServerAutoDetect();
		if (m_SPIT_Client != NULL) {
			m_ServerProjectName = p_DetectedServer.getProjectName();
			m_ServerIP = p_DetectedServer.getIP();

#if defined(DEBUGOUTPUT)
			printf("MySPITApp: connect to %s ", p_DetectedServer.getIP().data());
#endif
			//Use the m_BoardNumber as login name
			string stringLoginName = "";
			string stringLoginPassword = "";
			boolSuccess = m_SPIT_Client->connect(p_DetectedServer.getIP(), p_DetectedServer.getPort(), p_DetectedServer.getClientIP(), stringLoginName, stringLoginPassword); //ClientPort = "0" meeans use any port
		}
		if (boolSuccess == false) {
			//connection message could not be sent
			//perhaps restart AutoDetection
			startServerAutoDetect();
		}
		else {
			//WAITING FOR CONNECTION ANSWER
			//If the Server is closed in the meantime no answer will return
			//-> connect(..) has started the watchdog			
		}
		
//	} //Excample
}


/**
 * Is always called when a SPIT server does not respond anymore.<br>
 * For this you have to call SPIT_Client::startServerAutoDetect.<br>
 * This is an information used by autodetection and not by an established connection.<br>
 * <br>
 * When a connection to a SPITServer is established, the autodetection should be stopped.<br>
 * The connection lost event is detected by eventConnectionStatus<br>
 * <br>
 * @param p_DetectedServer a reference to a copy of a SAD_DetectedServer instance.<br>
 * After this function call the copy of the SAD_DetectedServer will be deleted!!<br>
 */
void MyApp::eventServerLost(const SAD_DetectedServer& p_DetectedServer) {
#if defined(DEBUGOUTPUT)
	printf("\n");
	printf("-----> SERVER LOST\n");
	printf("Name: %s  IP: %s\n", p_DetectedServer.getName().data(), p_DetectedServer.getIP().data());
	printf("\n");
#endif
}

void MyApp::eventServerChanged(const SAD_DetectedServer& p_DetectedServer) {
#if defined(DEBUGOUTPUT)
	printf("\n");
	printf("-----> SERVER LOST\n");
	printf("Name: %s  IP: %s\n", p_DetectedServer.getName().data(), p_DetectedServer.getIP().data());
	printf("\n");
#endif
	tryConnection(p_DetectedServer);
}

//---------------------------------------
//----------  Connecting event ----------

void MyApp::eventConnectionStatus(string p_ServerID, string p_ServerComputerName, string p_ServerName, int p_ServerStepTime, int p_ConnectionStatus, int p_OldConnectionStatus) {
#if defined(DEBUGOUTPUT)
	printf("MySPITApp: eventConnectionStatus %d ", p_ConnectionStatus);
#endif
	switch(p_ConnectionStatus) {
		case SPIT_Client::CONNECTION_NONE:
#if defined(DEBUGOUTPUT)
			printf(" not connected\n");
			//enter code 
#endif
			break;
			
		case SPIT_Client::CONNECTION_NONESERVER:
#if defined(DEBUGOUTPUT)
			printf(" connection terminated by server\n");
#endif
			//call eventProjectClosed because this may not be sended by the server,
			//when the connstion is closed
			eventProjectClosed_Server(m_ServerProjectID);

			
			//Server has terminated the connection -> startServerAutoDetect
			startServerAutoDetect();
			break;
			
		case SPIT_Client::CONNECTION_TRYCONNECTION:
		case SPIT_Client::CONNECTION_OK:
#if defined(DEBUGOUTPUT)
			printf(" connection established\n");
#endif
			stopServerAutoDetect(); //normally you should stop the autodetection before you try a connection
									//This is redundand
			m_ServerID = p_ServerID;
			m_ServerComputerName = p_ServerComputerName;
			m_ServerName = p_ServerName;
			m_ServerStepTime = p_ServerStepTime;
		
			//Here the connection is established -> don't reconnect at eventServerDetected(...)
			if (SPIT_Client::CONNECTION_OK == p_ConnectionStatus) {
				//connected
			}
			
			else {
				//reconnected
			}
			break;
			
		case SPIT_Client::CONNECTION_LOST:
#if defined(DEBUGOUTPUT)
			printf(" connection lost\n");
#endif
			//Maybe you want to reconnect, see eventServerDetected(...)
			//We have lost the connection -> start the autodetection to reconnect at eventServerDetected(...)
			startServerAutoDetect();
			break;
	}
	
}


//-----------------------------------------------------------
//---------- Server's Project events ------------------------
/**
 * The Server reports his loaded project
 * Called when<br>
 * a) the connection to the server is established.<br>
 * b) server load a new project<br>
 * c) server saves the project under a new name<br> 
 * @param p_ProjectID the server project's id<br>
 * (maybe the filename of the project or a part o it or the same as p_ProjectName)
 * @param p_ProjectName the server project's name
 * @param p_FramesPerSeoond All times in SPIT are given in frames. FramesPerSecond can be used to convert frames to seconds....
 */
void MyApp::eventReportProject_Server(string p_ProjectID, string p_ProjectName, int p_FramesPerSeoond) {
#if defined(DEBUGOUTPUT)
	printf("\n");
	printf("MySPITApp: eventProjectReport ID: %s Name: %s FramesPerSecond: %d\n", p_ProjectID.data(), p_ProjectName.data(), p_FramesPerSeoond);
#endif
	//set the server's ProjectID and ProjectName
	m_ServerProjectID = p_ProjectID;
	m_ServerProjectName = p_ProjectName;
	
	//set the frames per second as the server - perhaps the client's time values have to be recalculated. 
	m_ServerFramesPerSecond = p_FramesPerSeoond;
	
	//When the this client's project depends on the server's project -> load some specific informations here

}
/**
 * Called when the server closes his project.<br>
 * @param p_ProjectID the id of the project<br>
 * The server loads only one project at time-> the ID is not actually necessary
 */
void MyApp::eventProjectClosed_Server(string p_ProjectID) {
//	GPIO_Action::stopAllActions();
	//If we don't save SPITServer's project specific information
	//perhaps close all SPITObjects and RemoteSPITObjects to avoid conflicts, when the SPITServer loads a new project
//	GPIO_RemoteInput::eventSPITProjectClosed();
	SPITType::eventProjectClosed();
	SPITObject::eventProjectClosed();

	
#if defined(DEBUGOUTPUT)
	printf("MySPITApp: eventProjectClosed ID: %s\n", p_ProjectID.data());
	printf("\n");
#endif
}

/**
 * The server starts the transfer of its project, that means:.<br>
 * All SPITTypes and SPITObjects available on the server are transferred in the following.<br>
 * Several calls of eventSPITType and eventSPITObject will occur.<br>
 * At the end eventEndProjectTransfer_server is called.
 */
void MyApp::eventStartProjectTransfer_Server() {
#if defined(DEBUGOUTPUT)
	printf("MySPITApp: eventStartProjectTransfer\n");
#endif
}
/**
 * The transfer of the server's SPITTypes and SPITObjects is finished.<br>
 * AFter this eventSPITType and eventSPITObject occurs when types or objects are created or changed.
 */
void MyApp::eventEndProjectTransfer_Server() {
#if defined(DEBUGOUTPUT)
	printf("MySPITApp: eventEndProjectTransfer\n");
#endif
//	GPIO_Action::loadUserSettings_ProjectDepended();
}

/**
 * The client starts the transfer of its project, that means:.<br>
 * All SPITTypes and SPITObjects available on the client are transferred to the server.<br>
 * Several calls of confirming eventSPITType and eventSPITObject will occur, when the srver confirms the SPITTypes and SPITObjects.<br>
 * At the end eventEndProjectTransfer_Client is called.
 */
void MyApp::eventStartProjectTransfer_Client() {
#if defined(DEBUGOUTPUT)
	printf("MySPITApp: eventStartProjectTransfer\n");
#endif
}
/**
 * The transfer of the Client's SPITTypes and SPITObjects is finished.<br>
 */
void MyApp::eventEndProjectTransfer_Client() {
#if defined(DEBUGOUTPUT)
	printf("MySPITApp: eventEndProjectTransfer\n");
#endif
	//All SPITTypes (IC2_ActionInput, IC2_ActionOutput) and SPITObjects (corresponding objects for all IC2_Inputs and IC2_Outputs, creted automatically) 
	//are sent to the server
	
	//Send the actual values
		
}



//----------------------------------------------------
//---------       SPITType events           ----------

// SPITTypes describing an action or a type of an action.
// To trigger an action SPITObjects are used. Every SPITObject has one of the SPITTypes assigned.<br>
// There are two ways action can be triggered, either the server triggers an action on the client, 
// or the client sends remote control commands to the server. 
// a) normal types that are comitted to the client and played per SPITObjects by server, so the client can react with actions.
// b) Remote types are used for remote control of the server and are played by the client via SPIT objects.
//All normal SPITTypes are listet in a static unordered_map in the SPITType class.
//All remote SPITTypes are listet in a seperate static unordered_map in the SPITType class.
//You can use the SPITType::get... functions to deal with them.
//The SPITType class has a static recursive_mutex, you get a pointer of the mutex by: SPITType::getMutex();
//To be thread save you should cover your code with a lock sequence:
/*	
	{ //start of the lock sequence
		unique_lock<recursive_mutex> o_LockType(*SPITType::getMutex());
		//got the lock
		
		//fetch SPITType and deal with, but don't waste time while you holding the lock
		//Excample
		SPITType* ptrSPITType;
		for (int i = 0; i < SPITType::getSPITTypeCount(); i ++) {
			ptrSPITType = SPITType::getSPITType(i);
			if (ptrSPITType == NULL) break;
			//Do something with ptrSPITType
			//bew aware after you leave the lock sequence the ptrSPITType is no longer save!!
		}
	} //end of the lock sequence
 
	//For Remote Types you can use the getSPITTypeRemote... functions
*/

/**
 * The SPITCLientInterface (implemented by this) will be asked to veto the SPITType while SPIT_Client.parseReportSPITType.<br>
 * @param p_ReportSPITType
 * @return false = the SPITType is denied, true = the SPITType is accepted
 */
bool MyApp::checkSPITTypeVeto(SPIT_Message::ReportSPITType* p_ReportSPITType) {
	if (p_ReportSPITType == NULL) return true;
	SPITType* ptrSPITType;
	ptrSPITType = SPITType::getSPITType(p_ReportSPITType->getID());

	//Don't allow foreign SPITTypes 
	if (ptrSPITType == NULL) return false;
	if (ptrSPITType->getCreatedByClient() == false) return false;
	return true;
}

/**
 * Called when the server has created, loaded or changed a SPITType.<br>
 * May also occur when the server confirms a SPITType created or changed by the client.<br>
 * @param p_New true = server created a new SPITType<br>
 * false = server changed a existing SPITType<br>
 * @param p_SPITType a reference to a copy of the SPITType instance.<br>
 * The reference is deleted after this function call!! 
 */
void MyApp::eventSPITType(bool p_New, const SPITType& p_SPITType) {
	//fetch information of the SPITType
	if (p_New) {
		//a new SPITType is created
#if defined(DEBUGOUTPUT)
		printf("MySPITApp: eventSPITType NEW");
#endif
	}
	else {
		//an existing SPITType is changed
#if defined(DEBUGOUTPUT)
		printf("MySPITApp: eventSPITType CHANGED");
#endif
	}
#if defined(DEBUGOUTPUT)
	printf(" ID: %s GroupName: %s Name: %s confirmed: %d\n", p_SPITType.getID().data(), p_SPITType.getGroupName().data(), p_SPITType.getName().data(), p_SPITType.getConfirmed());
#endif

	//Check your actions
	//Whenever you provide an action for this SPITType take this over in your code.
	//call m_SPIT_Client->setActionSPITType(p_SPITType.getID(), true);
	
	//for our little excample as described in loadClientProject we do following
	if (m_SPIT_Client != NULL) {
		//This may be redundand, because we defined the SPITTypes as fixed, so the server can't change them
		//But foe all SPITTypes which are not fixed we need to setActionSPITType
		if ("MySPITAPP_Action1ID" == p_SPITType.getID()) {
			m_SPIT_Client->setActionSPITType(p_SPITType.getID(), true);
		}
		if ("MySPITAPP_Action2ID" == p_SPITType.getID()) {
			m_SPIT_Client->setActionSPITType(p_SPITType.getID(), true);
		}
	}
}
/**
 * Called when the server has created or changed a Remote SPITType.<br>
 * Remote SPITType are used in Remote SPITObjects. A Remote SPITObject is used to remote control the SPITServer.
 * @param p_New true = server created a new SPITType<br>
 * false = server changed a existing SPITType<br>
 * @param p_SPITType a reference to a copy of the SPITType instance.<br>
 * The reference is deleted after this function call!! 
 */
void MyApp::eventSPITTypeRemote(bool p_New, const SPITType& p_SPITType) {
	if (p_New) {
#if defined(DEBUGOUTPUT)
		printf("MySPITApp: eventSPITTypeRemote NEW");
#endif
	}
	else {
		//inform all Remote SPITObjects using the p_SPITType
		//the SPITObject uses information of the SPITType
		
		SPITObject* ptrSPITObject;
		int intSPITObjectCount;
		intSPITObjectCount = SPITObject::getSPITObjectRemoteCount();
		for (int intObject = 0; intObject < intSPITObjectCount; intObject++) {
			ptrSPITObject = SPITObject::getSPITObjectRemote(intObject);
			if (ptrSPITObject == NULL) continue;
			if (ptrSPITObject->getSPITTypeID() == p_SPITType.getID()) {
				//call ecentSPITObjectRemote	
				eventSPITObjectRemote(false, p_SPITType, *ptrSPITObject);
			}
		}
#if defined(DEBUGOUTPUT)
		printf("MySPITApp: eventSPITTypeRemote CHANGED");
#endif
	}
#if defined(DEBUGOUTPUT)
	printf(" ID: %s GroupName: %s Name: %s confirmed: %d\n", p_SPITType.getID().data(), p_SPITType.getGroupName().data(), p_SPITType.getName().data(), p_SPITType.getConfirmed());
#endif
	//this is a remote SPITType, if you want you can list this for later remote use. 
}

/**
 * Called when the server has removed a SPITType.<br>
 * After this function has been exited the SPITType is emeoved from the SPITType-List and the pointer is deleted.<br>
 * So use the pointer immediately or use the copyconstructor to make a copy for extended use.
 * You have to release the copy yourself later (delete ptr...)
 * @param p_SPITType a reference to a copy of the SPITType instance.<br>
 * The reference is deleted after this function call!! 
 */
void MyApp::eventSPITTypeRemoved(const SPITType& p_SPITType) {
//	GPIO_Action* ptrGPIO_Action;
//	ptrGPIO_Action = GPIO_Action::getGPIO_Action(p_SPITType.getID());
//	if (ptrGPIO_Action != NULL) {
//		ptrGPIO_Action->invoke("", false, 0, 0, false, true, 0);
//	}
	
#if defined(DEBUGOUTPUT)
	printf("MySPITApp: eventSPITType Removed");
	printf(" ID: %s GroupName: %s Name: %s\n", p_SPITType.getID().data(), p_SPITType.getGroupName().data(), p_SPITType.getName().data());
#endif

	//Check your actions
	//take the removal of the SPITType over in your code
	
	//CAUTION: Deleted SPIT types can be added later again (perhaps per UNDO on the SPITServer)
	//-> don't remove your assigned action immediately - keep it to reassign it later	
}
/**
 * Called when the server has removed a Remote SPITType.<br>
 * Remote SPITType are used in Remote SPITObjects. A Remote SPITObject is used to remote control the SPITServer.
 * After this function has been exited the SPITType is emeoved from the Remote SPITType-List and the pointer is deleted.<br>
 * So use the pointer immediately or use the copyconstructor to make a copy for extended use.
 * You have to release the copy yourself later (delete ptr...)
 * @param p_SPITType a reference to a copy of the SPITType instance.<br>
 * The reference is deleted after this function call!! 
 */
void MyApp::eventSPITTypeRemoteRemoved(const SPITType& p_SPITType) {
#if defined(DEBUGOUTPUT)
	printf("MySPITApp: eventSPITTypeRemote Removed");
	printf(" ID: %s GroupName: %s Name: %s\n", p_SPITType.getID().data(), p_SPITType.getGroupName().data(), p_SPITType.getName().data());
#endif

	//This is a remote SPITType
	//If you have listet this prior you should remove it from the list
}




//----------------------------------------------------
//---------       SPITObject events         ----------
// Every SPITObjects has one one SPITType assigned.
// Whenever an action is to be triggered, play events are sent with the corresponding SPITObject. 
// All SPITObjects are listet in a static unordered_map in the SPITObject class.
// All Remote SPITObjects are listet in a seperate unordered_map in the SPITObject class;
// You can use the SPITObject::get... functions to deal with them.
// The SPITObject class has a static recursive_mutex, you get a pointer of the mutex by: SPITObject::getMutex();
// To be thread save you should cover your code with a lock sequence:
/*	
	{ //start of the lock sequence
		unique_lock<recursive_mutex> o_LockObject(*SPITObject::getMutex());
		//got the lock
		
		//fetch SPITObjects and deal with, but don't waste time while you holding the lock
		//Excample
		SPITObject* ptrSPITObject;
		for (int i = 0; i < SPITObject::getSPITObjectCount(); i++) {
			ptrSPITObject = SPITObject::getSPITObject(i);
			if (ptrSPITObject == NULL) break;
			//Do something with ptrSPITObject
			//bew aware after you leave the lock sequence the ptrSPITObject is no longer save!!
		}
	} //end of the lock sequence
  
	//For Remote Objects you can use the getSPITObjectRemote... functions
*/

/**
 * The SPITCLientInterface (implemented by this) will be asked to veto the SPITObject while SPIT_Client.parseReportSPITObject.<br>
 * @param p_ReportSPITObject
 * @return false = the SPITType is denied, true = the SPITType is accepted
 */
bool MyApp::checkSPITObjectVeto(SPIT_Message::ReportSPITObject* p_ReportSPITObject) {
	if (p_ReportSPITObject == NULL) return true;
	SPITType* ptrSPITType;
	ptrSPITType = SPITType::getSPITType(p_ReportSPITObject->getSPITTypeID());
	if (ptrSPITType == NULL) return false;
	
	//Don't allow SPITObjects which refers to a SPITType which is not created by this client - other SPITTypes cannot be handled here
	//if (ptrSPITType->getCreatedByClient() == false) return false;

	//	SPITObject* ptrSPITObject;
//	ptrSPITObject = SPITObject::getSPITObject(p_ReportSPITOnject->getID());
	//Don't allow foreign SPITTypes 
//	if (ptrSPITObject == NULL) return false;
	
	return true;
	
}

/**
 * Called when the server has created, loaded or changed a SPITObject.<br>
 * May also occur when the server confirms a SPITObject created by the client, what the client should never do!!.<br>
 * @param p_New true = server created a new SPITObject<br>
 * false = server changed a existing SPITObject<br>
 * @param p_SPITType a reference to a copy of the associated SPITType instance.<br>
 * The reference is deleted after this function call!! 
 * @param p_SPITObject a reference to a copy of the SPITObject instance.<br>
 * The reference is deleted after this function call!! 
 */
void MyApp::eventSPITObject(bool p_New, const SPITType& p_SPITType, const SPITObject& p_SPITObject) {
	if (p_New) {
		int intTest = 0;
#if defined(DEBUGOUTPUT)
		printf("MySPITApp: eventSPITObject NEW");
#endif
	}
	else {
#if defined(DEBUGOUTPUT)
		printf("MySPITApp: eventSPITObject CHANGED");
#endif
	}	
#if defined(DEBUGOUTPUT)
	printf(" ID: %s Name: %s -> SPITType: %s\n", p_SPITObject.getID().data(), p_SPITObject.getName().data(), p_SPITType.getName().data());
#endif
	

/*	
	//to list all SPITObjects use 
	SPITObject* ptrSPITObject;
	int intSPITObjectCount = SPITObject::getSPITObjectCount();
	for (int intObject = 0; intObject < intSPITObjectCount; intObject++) {
		ptrSPITObject = SPITObject.getSPITObject(intObject);
		if (ptrSPITObject == NULL) continue;
		//Do something with ptrSPITObject
		
	}
*/	
	
	
	
	//check your actions
	//Whenever you provide an action for this SPITObject take this over in your code.
	//call m_SPIT_Client->setActionSPITObject(p_SPITObject.getID(), true);
	
	//for our little excample as described in loadClientProject:
	//We have only actions for SPITTypes - means every SPITObject with the associated "Actionx"-SPITTypes will trigger the action when it is played
	//-> no need to do something here, because setActionSPITType has been set prior and the SPITObject will automatically inherits this
	//SPITObjects with other SPITTypes asscciated will trigger no actions
	
}
/**
 * Called when the server has created or changed a Remote SPITObject.<br>
 * A Remote SPITObject is used to remote control the SPITServer.
 * @param p_New true = server created a new remote SPITObject<br>
 * false = server changed a existing remote SPITObject<br>
 * @param p_SPITType a reference to a copy of the associated SPITType instance.<br>
 * The reference is deleted after this function call!! 
 * @param p_SPITObject a reference to a copy of the SPITObject instance.<br>
 * The reference is deleted after this function call!! 
 */
void MyApp::eventSPITObjectRemote(bool p_New, const SPITType& p_SPITType, const SPITObject& p_SPITObject) {
	if (p_New) {
#if defined(DEBUGOUTPUT)
		printf("MySPITApp: eventSPITObjectRemote NEW");
#endif
	}
	else {
#if defined(DEBUGOUTPUT)
		printf("MySPITApp: eventSPITObjectRemote CHANGED");
#endif
	}
#if defined(DEBUGOUTPUT)
	printf(" ID: %s Name: %s -> SPITType: %s\n", p_SPITObject.getID().data(), p_SPITObject.getName().data(), p_SPITType.getName().data());
#endif
	
	//This is a remote SPITObject which can be used to remote control the server.
	//If you want you can list this for later use
	
//	GPIO_RemoteInput::eventSPITObjectRemote(p_New, p_SPITType, p_SPITObject);
}

/**
 * Called when the server has removed a SPITObject.<br>
 * After this function has been exited the SPITObject is emeoved from the SPITObject-List and the pointer is deleted.<br>
 * So use the pointer immediately or use the copyconstructor to make a copy for extended use.
 * You have to release the copy yourself later (delete ptr...)
 * @param p_SPITType a reference to a copy of the associated SPITType instance.<br>
 * The reference is deleted after this function call!! 
 * @param p_SPITObject a reference to a copy of the SPITObject instance.<br>
 * The reference is deleted after this function call!! 
 */
void MyApp::eventSPITObjectRemoved(const SPITType& p_SPITType, const SPITObject& p_SPITObject) {
#if defined(DEBUGOUTPUT)
	printf("MySPITApp: eventSPITObject Removed");
	printf(" ID: %s Name: %s\n", p_SPITObject.getID().data(), p_SPITObject.getName().data());
#endif
	
	//Check your actions
	//take the removal of the SPITObject over in your code
	
	//CAUTION: Deleted SPITObjects can be added later again (perhaps per UNDO on the SPITServer)
	//-> don't remove your assigned action immediately - keep it to reassign it later	
}
/**
 * Called when the server has removed a Remote SPITObject.<br>
 * A Remote SPITObject is used to remote control the SPITServer.
 * After this function has been exited the SPITObject is emeoved from the SPITObject-List and the pointer is deleted.<br>
 * So use the pointer immediately or use the copyconstructor to make a copy for extended use.
 * You have to release the copy yourself later (delete ptr...)
 * @param p_SPITType a reference to a copy of the associated SPITType instance.<br>
 * The reference is deleted after this function call!! 
 * @param p_SPITObject a reference to a copy of the SPITObject instance.<br>
 * The reference is deleted after this function call!! 
 */
void MyApp::eventSPITObjectRemoteRemoved(const SPITType& p_SPITType, const SPITObject& p_SPITObject) {
#if defined(DEBUGOUTPUT)
	printf("MySPITApp: eventSPITObjectRemote Removed");
	printf(" ID: %s Name: %s\n", p_SPITObject.getID().data(), p_SPITObject.getName().data());
#endif
	
	//This is a remote SPITObject
	//If you have listet this prior you should remove it from the list
//	GPIO_RemoteInput::eventSPITObjectRemoteRemoved(p_SPITType, p_SPITObject);
}




//--------------------------------------------------------------
//----------------       Playing SPITObjects    ----------------
// Following is called when the SPITServer play SPITObjects.
// For every step the server does following sequence occurs:
// a) eventPlay
// b) several eventPlayObject for each actual played SPITObjectin this step
// NEXT STEP of the server
// a) eventPlay
// b) several eventPlayObject for each actual played SPITObject in this step
// ...

// SPITServer may use several methods for playing SPITObjects.
// e.g. liveSHOWsoftware has the timeline and jingles which plays objects, where the timeline the main player is.
// SPIT distinguishes between the different players. All eventPlay... functions contain the information which player plays the SPIT object.

void MyApp::eventPlay(string p_PlayerID, string p_PlayerName, char p_PlayerType, bool p_PlayerIsRunning, long long p_TimeFrames) {
#if defined(DEBUGOUTPUT)
	printf("MySPITApp: eventPlay Player: %s Runs: %d Time: %lld\n", p_PlayerName.data(), p_PlayerIsRunning, p_TimeFrames);
#endif
	
	switch(p_PlayerType) {
		case SPIT_Client::PLAYERTYPE_MAIN:
			//p_FrameTimes contains the position inside the server's project, we can use it like a timecode
//			int intMilliseconds;
//			intMilliseconds= (int)(((double)p_TimeFrames / (double)m_ServerFramesPerSecond) * 1000);
			break;
		case SPIT_Client::PLAYERTYPE_CONTINOUS:
			//The olayer is playing only single Object sequences sometime during the project
			break;
		case SPIT_Client::PLAYERTYPE_DISCRETE:
			//Only discrete values where submitted, e.g the fader position of remote faders.
			break;
	}
}



/**
 * Called when the SPITServer plays a SPITObject. For every played SPITObject one eventPlayObject is called.<br>
 * You can decide here which action should be done or not.<br>
 * 
 * @param p_Started true = The SPITObject is newly played (e.g. in liveSHOWsoftware the cursor enters the SPITObject)
 * @param p_Ended Das true = Playback of SPITObject finished (e.g. in liveSHOWsoftware the cursor has exited the SPIT object)
 * @param p_SPITTypeID the ID of the associated SPITType of p_SPITObject
 * @param p_PlayerID the id of the player
 * @param p_PlayerIsRunning whether the player is running or not<br>
 */
void MyApp::eventPlaySPITType(string p_PlayerID, string p_SPITTypeID, bool p_PlayerIsRunning, bool p_Started, bool p_Ended, double p_Value) {
	//enter code
}




//--------------------------------------------------------------
//----------------   Remote Control SPITServer  ----------------
// The SPITServer has sent all possibilities to control it remotely - remote SPITObjects - see eventSPITObjectRemote
// Some of them may control functionality with values (fader psotions for loudness of sound, ...).
// When the server changes one of these value objects, it sends an eventPlayRemoteObjectStatus.

/**
 * Called when a SpitObject's playstatus (frameposition, value) is changed by the server.<br>
 * If p_SPITObject is a remote object -> This allows the Client to adjust its display for the remote command.<br> 
 * If p_SPITObject is NOT a remote object -> The server has changed a value without the player sequence.<br> 
 * @param p_SPITType the assigned remote SPITType of p_SPITObject
 * @param p_SPITObject the remote SPITObject which is changed
 * @param p_FrameInsideObject can be discarded
 * @param p_Value the new value
 */
void MyApp::eventSPITObjectPlayStatus(const SPITType& p_SPITType, const SPITObject& p_SPITObject, long long p_FrameInsideObject, int p_Value) {
#if defined(DEBUGOUTPUT)
	printf("MySPITApp: eventPlayRemoteObjectStatus RemoteSPITType: %s RemoteSPITObject: %s Position: %lld Value: %d\n", p_SPITType.getName().data(), p_SPITObject.getName().data(), p_FrameInsideObject, p_Value);
#endif
	if (p_SPITObject.getServerRemote() == true) {
		
	}
	else {
		
	}
}

/**
 * Send a remote control command to the SPITServer.<br>
 * There may be different command types:
 * a) Some don't use value e.g. go to next scene, olay a jingle, ..
 * b) Some use values e.g. the main sound fader
 * @param p_SPITObjectID the id of the remote SPITObject (this represents the remote control command)
 * @param p_Valueppm the value in ppm <br>
 * in case a) the value is ignored<br>
 * in case b) the value is used
 * @param p_ValueFlag indicates how the server should use p_Valueppm:<br>
 * VALUEFLAG_DIRECT (0x00) The server will overwrite his existing value with p_Value<br>
 * VALUEFLAG_INCREASE (0x01) The server increases his existing value by p_Valueppm
 * VALUEFLAG_DECREASE (0x01) The server decreases his existing value by p_Valueppm
 */
void MyApp::sendRemoteObject(string p_SPITObjectID, int p_Valueppm, char p_ValueFlag) {
	if (m_SPIT_Client != NULL) {
		m_SPIT_Client->sendRemoteObject(p_SPITObjectID, p_Valueppm, p_ValueFlag);
	}
}

/*void MyApp::testLiveShowRemote() {
	//We jump to the next Scenebridge - no Value is needed
	m_SPIT_Client->sendRemoteObject("nextSceneBridge");
	//we set the main sound volume to 0.1, SPIT uses ppm -> 0.1 = 100000
	m_SPIT_Client->sendRemoteObject("mainmixerSound_Volume00", 100000, SPIT_Client::VALUEFLAG_DIRECT);
	//we increase the main sound volume by 0.1, SPIT uses ppm -> 0.1 = 100000
	m_SPIT_Client->sendRemoteObject("mainmixerSound_Volume00", 100000, SPIT_Client::VALUEFLAG_INCREASE);
}
*/


//-------------------------------------------------
//---------- Options for the client user ----------
//Normally this should not be used, because because the server is master of things
//In some situations, however, it may be helpful

/**
 * Removes the SPITType with id == p_SPITTypeID
 * @param p_SPITTypeID the id of the SPITType
 * @return 
 */
int MyApp::removeSPITType(string p_SPITTypeID) {
	if (m_SPIT_Client == NULL) return -1;
	return  m_SPIT_Client->removeSPITType(p_SPITTypeID);
}
/**
 * Removes the SPITObject with id == p_SPITObjectID
 * @param p_SPITObjectID the id of the SPITObject
 * @return 
 */
int MyApp::removeSPITObject(string p_SPITObjectID) {
	if (m_SPIT_Client == NULL) return -1;
	return m_SPIT_Client->removeSPITObject(p_SPITObjectID);
}
/**
 * Changes the SPITType 
 * @param p_SPITType a pointer to a SPITType instance whith changed members
 * @return 
 */
int MyApp::changeSPITType(SPITType* p_SPITType) {
	return  m_SPIT_Client->changeSPITType(p_SPITType);
}
/**
 * Changes the SPITObject
 * @param p_SPITObject a pointer to a SPITObject with changed members
 * @return 
 */
int MyApp::changeSPITObject(SPITObject* p_SPITObject) {
	return m_SPIT_Client->changeSPITObject(p_SPITObject);
}


